(defmacro show-env [] (println (keys &env)) `(println ~@(keys &env)))


(defn java?
  "Is the symbol a Java method?"
  [sym]
  (or (re-find #"^\." (str sym))    ;; starts with ".", e.g. .indexOf
      (and (re-find #"/" (str sym)) ;; e.g. Math/sqrt
           (re-find #"Class"
                    (-> sym str (str/split #"/") first symbol resolve type str)))))

(defn non-java-fn
  "get the function part of a predicate IF it's not java, else dummy"
  [pred]
  (if (java? (first pred))
    str (first pred)))

(defn good-fn
  "get the function part of a predicate IF it's not java, else dummy"
  [pred]
  (if (try (resolve (first pred)) (catch Exception e nil))
    (first pred) str))


(defmacro gen? [pred]
  ;  `(v/with-logic-vars) -- may need this, but ok for now
  `(plat/gen? ~(if (resolve (first pred)) (first pred) str)))




(defmacro exp-query [outvars & predicates]
  "Shows operations and predicates. Tweak of expand-query to
  return a value instead of printing.

  Syntax: (expand-query outvars & predicates)

  Ex: (expand-query [?person] (age ?person 25))"
  `(v/with-logic-vars
     (let [{outvars# :output-fields
            predicates# :predicates}
           (parse/prepare-subquery ~outvars
                                   [~@(map vec predicates)])]
       (parse/build-query outvars# predicates#))))


(defmacro exp-query-str [outvars & predicates]
  "Shows operations and predicates. Tweak of expand-query to
  return a value instead of printing.

  Syntax: (expand-query outvars & predicates)

  Ex: (expand-query [?person] (age ?person 25))"
  `(v/with-logic-vars
     (let [{outvars# :output-fields
            predicates# :predicates}
           (parse/prepare-subquery ~outvars
                                   [~@(map vec predicates)])]
       (-> (parse/build-query outvars# predicates#)
           pr-str (clojure.core/read-string) rest rest ) )))


(defn has-casca
  "does form contain a cascalog var?"
  [form]
  (or (and (symbol? form)
           (v/cascalog-var? form))
      (and (seq? form)
           (some has-casca form) )))

(defn has-nested-casca
  "does the form contain a nested cascalog var?"
  [form]
  (let
      [sub-forms (filter seq? form)]
    (some has-casca sub-forms)))




(defn anonymize-java
  "transform predicate so first element is anonymized if it's java"
  [pred]
  (if-not (java? (first pred))
    pred
    (let
        [pre-op (take-while #(not (v/selector? %) ) pred)
         post-op (drop-while #(not (v/selector? %)) pred)]
      `(~(lambda (first pred) (dec (count pre-op)))
        ~@(rest pre-op)
        ~@post-op))))


(defn anonymize-all
  "transform predicate so all funcs are anonymized
   TODO"
  [pred]
  (if-not (java? (first pred))
    pred
    (let
        [pre-op (take-while #(not (v/selector? %) ) pred)
         post-op (drop-while #(not (v/selector? %)) pred)]
      `(~(lambda (first pred) (dec (count pre-op)))
        ~@(rest pre-op)
        ~@post-op))))

(defmacro normalize-pred [pred]
  `(v/with-logic-vars
     (-> (parse/prepare-subquery nil [~(vec pred)] )
         :predicates first pr-str read-string)))

(defmacro normalize-preds [preds]
  `(v/with-logic-vars
     (-> (parse/prepare-subquery nil [~@(map vec preds)] )
         :predicates pr-str (clojure.core/read-string))))

(defn prepare-preds [predicates]
  `(normalize-preds ~(map anonymize-java predicates)))

(defmacro ??<< [outvars & predicates]
  ;;  (prepare-preds predicates)
  `(??<- ~outvars
         ~@(normalize-preds ~(map anonymize-java predicates)))
  ;; `(??<- ~outvars
  ;;        ~(prepare-preds predicates))
  )


(defmacro transform-preds [predicates]
  (concat (map clause-to-rpn predicates)
          (get-casca-read-nums predicates)))





(defmacro maybe-to-rpn
  "convert to rpn where possible"
  [pred]
  (if (v/with-logic-vars (plat/gen? (non-java-fn ~pred)))
    "gen" ;; (v/with-logic-vars ~pred)
    "not" ;; (v/with-logic-vars (clause-to-rpn ~pred))
    )
  )

(defn detect-gen [pred]
  `(maybe-to-rpn ~pred))




(defmacro preds-to-rpn
  "convert to rpn where possible"
  [outvars & preds]
  (concat `(??<- ~outvars)
          [(if (v/with-logic-vars (plat/gen? (non-java-fn (first preds))))
             (v/with-logic-vars (first preds))
             (v/with-logic-vars (clause-to-rpn (first preds))))
           (if (v/with-logic-vars (plat/gen? ~(non-java-fn (second preds))))
             (identity 1 :> ?z);; (v/with-logic-vars ~pred)
             (v/with-logic-vars (clause-to-rpn ~(second preds))))]))

(defmacro ??<< [outvars & predicates]
  `(v/with-logic-vars
     (??<- ~outvars
           ~@(map #(if (plat/gen? (non-java-fn %))
                     %
                     (clause-to-rpn %))
                  predicates)
           ~@(get-casca-read-nums predicates))))
