(ns pallet.crate.ssl
  "SSL Keystore management"
  (:use
   [pallet.stevedore :only [script map-to-arg-string]]
   [pallet.utils :only [cmd-join]]))

(defn ssl-genkey-jks*
  "Create an SSL keystore using java.  Creates a JKS format keystore.
     alias             - alias used for accessing the key
     store-password    - the password for the keystore (at least 6 characters)
     full-name         - your first and last names (CN)
     organisation-unit - unit within your organisation (OU)
     organisation      - your organisation (O)
     location          - your city or region (L)
     state             - your state or province (ST)
     country-code      - two letter country code (C)

   Recognised options:
     :keystore path  - path to the keystore to be used
     :keyalg RSA|DSA  - key algorithm, defaults to RSA.
     :keypass password - key password, defaults to store password"
  [alias store-password common-name organisation-unit organisation location
   state country-code & options]
  (let [options (apply hash-map options)
        options (merge {:keyalg "RSA"} options)]
    (script
     (keytool
      -genkey -alias ~alias -storepass ~store-password
      -dname
      ~(str
        "'CN=" common-name ",OU=" organisation-unit ",O=" organisation
        ",L=" location ",S=" state ",C=" country-code "'")
      -keypass ~(get options :keypass store-password)
      ~(map-to-arg-string options :dash "-")))))

(defn ssl-import-chain-certificate-jks*
  "Import a chain certificate.
   Recognised options:
     :keystore path  - path to the keystore to be used"
  [alias chain-certificate & options]
  (let [options (apply hash-map options)]
    (script
     keytool -import -alias ~alias -trustcacerts -file ~chain-certificate
     ~(map-to-arg-string options :dash "-"))))

(defn ssl-import-certificate-jks*
  "Import a chain certificate.
   Recognised options:
     :keystore path  - path to the keystore to be used"
  [alias chain-certificate & options]
  (let [options (apply hash-map options)]
    (script
     keytool -import -alias ~alias -file ~chain-certificate
     ~(map-to-arg-string options :dash "-"))))

(defn ssl-import-pkcs12*
  "Import a pkcs12 ket.  Uses openssl."
  [alias key-crt key-key cert-p12 ca-crt ca-name]
  (script
   (openssl pkcs12 -export -in ~key-crt -inkey ~key-key -out ~cert-p12
            -name ~alias -CAfile ~ca-crt -caname ~ca-name -chain)))
