(ns clj-scsynth.core
  (:import com.sun.jna.ptr.PointerByReference)
  (:use [clojure.contrib.java-utils :only [file]]
        [clojure.contrib.find-namespaces :only [find-namespaces-in-dir]]
        [leiningen.deps :only [deps]])
  (:use [clj-native.direct :only [defclib]]))

(defn get-by-pattern
  "Gets a value from map m, but uses the keys as regex patterns,
  trying to match against k instead of doing an exact match."
  [m k]
  (m (first (drop-while #(nil? (re-find (re-pattern %) k))
                        (keys m)))))

(def native-names
     {"Mac OS X" :macosx
      "Windows" :windows
      "Linux" :linux
      "SunOS" :solaris
      "amd64" :x86_64
      "x86_64" :x86_64
      "x86" :x86
      "i386" :x86
      "arm" :arm
      "sparc" :sparc})

(defn get-os
  "Returns a keyword naming the host OS."
  []
  (get-by-pattern native-names (System/getProperty "os.name")))

(defn get-arch
  "Returns a keyword naming the host architecture"
  []
  (get-by-pattern native-names (System/getProperty "os.arch")))

(defn find-native-lib-path
  "Returns a File representing the directory where native libs for the
  current platform are located."
  []
  (let [osdir (name (get-os))
        archdir (name (get-arch))
        f (file "native" osdir archdir)]
    (if (.exists f)
      f
      nil)))

(defn find-synthdefs-lib-path
  "Returns a File representing the directory where synthdefs libs for scsynth are located."
  []
  (let [osdir (name (get-os))
        archdir (name (get-arch))
        f (file "native" osdir archdir "synthdefs")]
    (if (.exists f)
      f
      nil)))

(defn find-scsynth-lib-path
  "Returns a File representing the directory where synthdefs libs for scsynth are located."
  []
  (let [osdir (name (get-os))
        archdir (name (get-arch))
        f (cond (= "linux" osdir) (file "native" osdir archdir "libscsynth.so")
                (= "windows" osdir) (file "native" osdir archdir "scsynth.dll"))]
    (if (.exists f)
      f
      nil)))

(System/setProperty "jna.library.path" (str (find-native-lib-path)))

(defclib
  scsynth
  (:functions 
   (World_WaitForQuit [void*] void)))

(defclib
  scsynth-jna
  (:structs
   (snd-buf
    :samplerate double
    :sampledur double
    :data void*
    :channels int
    :samples int
    :frames int
    :mask int
    :mask1 int
    :coord int
    :sndfile void*)
   (sc-jna-startoptions
    :udp-port-num int
    :tcp-port-num int    
    :verbosity int
    :lib-scsynth-path constchar*
    :plugin-path constchar*))
  (:functions 
   (ScJnaStart [sc-jna-startoptions*] void*)
   (ScJnaCleanup [] void)
   (ScJnaCopySndBuf [void* int] snd-buf*)))

(loadlib-scsynth)
(loadlib-scsynth-jna)
