(ns clj-sclang.core
  (:import com.sun.jna.ptr.PointerByReference)
  (:use [clojure.contrib.java-utils :only [file]]
        [clojure.contrib.find-namespaces :only [find-namespaces-in-dir]])

  (:use [clj-native.direct :only [defclib loadlib typeof]]
        [clj-native.structs :only [byref byval]]
        [clj-native.callbacks :only [callback]]))

(defn get-by-pattern
  "Gets a value from map m, but uses the keys as regex patterns,
  trying to match against k instead of doing an exact match."
  [m k]
  (m (first (drop-while #(nil? (re-find (re-pattern %) k))
                        (keys m)))))

(def native-names
     {"Mac OS X" :macosx
      "Windows" :windows
      "Linux" :linux
      "SunOS" :solaris
      "amd64" :x86_64
      "x86_64" :x86_64
      "x86" :x86
      "i386" :x86
      "arm" :arm
      "sparc" :sparc})

(defn get-os
  "Returns a keyword naming the host OS."
  []
  (get-by-pattern native-names (System/getProperty "os.name")))

(defn get-arch
  "Returns a keyword naming the host architecture"
  []
  (get-by-pattern native-names (System/getProperty "os.arch")))

(defn find-native-lib-path
  "Returns a File representing the directory where native libs for the
  current platform are located."
  []
  (let [osdir (name (get-os))
        archdir (name (get-arch))
        f (file "native" osdir archdir)]
    (if (.exists f)
      f
      nil)))

(System/setProperty "jna.library.path" (str (find-native-lib-path)))

(defclib
  sclang-jna
  (:callbacks
   (vpost-cb [constchar*] void))
  (:functions
   (schedInit [] void)   
   (init_OSC [int] void)   
   (pyr_init_mem_pools [int int] bool) 
   (schedRun [] void) 
   (schedStop [] void)   
   (compileLibrary [] bool)
   (runLibrary [void*] void)
   (scGlobals [] void*)
   (runInterpreter [void* void* int] void)

   (getsym [constchar*] void*)
   (findsym [constchar*] void*)

   (isCompiledOk [] bool)
   (setCmdLine [void* constchar*] void)

   (set_vpost_callback [vpost-cb] void)))

(loadlib sclang-jna)

(defmacro make-vpost-cb [cb-fn]
  `(callback vpost-cb ~cb-fn))

(defn sclang-start [cb]
  (set_vpost_callback cb)
  (pyr_init_mem_pools (* 1 1024 1024) (* 256 1024))
  (init_OSC 57120)
  (schedInit)
  (compileLibrary)
  (isCompiledOk))

(defn interpret [code] (let [s_inter (getsym "interpretCmdLine")
                             g (scGlobals)]
                         (setCmdLine g code)                         
                         (runLibrary s_inter)))

(defn sclang-connect-to-external [hostname port] 
  (interpret  (str "
n = NetAddr(\"" hostname "\", " port " );
n.connect;
Server.local = Server.default = s = Server(\\localhost, n);
s.remoteControlled = false;
s.boot;
s.serverRunning = true;
s.notify;
s.initTree;
")))

(comment
  (def cb (make-vpost-cb (fn [txt] (println txt))))
  (sclang-start cb)
  (interpret "\"Hello, World!\".postln")
  )