(ns mozinator.repl
  (:gen-class)
  (:import                  
   (javax.swing JFrame JEditorPane JScrollPane JPanel JTextField SwingUtilities)
   (java.io StringReader PushbackReader OutputStreamWriter PrintWriter)
   (java.util.concurrent LinkedBlockingQueue)
   (jsyntaxpane DefaultSyntaxKit)
   (java.awt Dimension BorderLayout)
   (clojure.lang IDeref))

  (:require [clojure.main :as r])
  
  (:use [hiredman.hydra :only (react-on event hydra)]
        [clojure.stacktrace :only (e)])
    
  (:use clojure.contrib.seq-utils))

;; Swing macros 

(defmacro EDT
  "runs body on the Event-Dispatch-Thread (Swing)"
  [& body]
  `(SwingUtilities/invokeLater (fn [] ~@body)))

(defmacro action-performed [component event & body]
  `(. ~component addActionListener
      (proxy [java.awt.event.ActionListener] []
        (actionPerformed [~event] ~@body))))

;; REPL functions

(defn queue->outputstream [jta renderfn]
  (let [buffer (StringBuffer.)]
    (proxy [java.io.OutputStream IDeref] []
      (deref [] buffer)
      (flush []
             (when (< 0 (.length buffer))
               (let [sb (.toString buffer)]
                 (renderfn jta sb)
                 (.setLength buffer 0))))
      (close [] (.flush this))
      (write
       ([i] (.append buffer (char i)))
       ([buf off len]
          (doseq [i (take len (drop off buf))]
            (.append buffer (char i))))))))

(defn send-to-repl
  [Q text]
  (future (Q (event ::read
                    (-> text
                        StringReader.
                        PushbackReader.)))))

(defn start-repl-thread [Q renderfn promptfn]
  (future
   (try
    (binding [*out* (-> Q (queue->outputstream renderfn) OutputStreamWriter. PrintWriter.)]
      (let [read-q (LinkedBlockingQueue.)]
        (react-on [Q to ::read as rdr] (.put read-q rdr))
        (clojure.main/repl
         :caught (fn [x]
                   (binding [*e x] (e))
                   (.flush *out*)
                   (println ""))
         :need-prompt (constantly true)
         :prompt #(do (promptfn Q (str (ns-name *ns*))))
         :read (fn [a b]
                 (binding [*in* (.take read-q)]
                   (r/repl-read a b))))))
    (catch Exception e
      (println (pr-str e))))))

(defn -main []
  (EDT
   (let [frame      (JFrame.)
         content    (.getContentPane frame)
         editor     (JEditorPane.)
         scroll     (JScrollPane. editor)
         panel      (JPanel.)
         prompt     (JTextField.)
         text-field (JTextField.)
         queue      (hydra)]      
     
     (DefaultSyntaxKit/initKit) 

     (doto queue
       (start-repl-thread (fn [q itm] (EDT                                              
                                       (let [ doc (.getDocument editor)
                                             length (.getLength doc)]
                                         (.insertString doc length itm nil))))
                          
                          (fn [q ns] (EDT
                                      (.setText prompt (str ns))))))

     (doto prompt
       (.setEditable false)
       (.setPreferredSize (Dimension. 150 20)))

     (doto text-field
       (action-performed event
                         (EDT
                          (send-to-repl queue (.getText text-field))
                          (.setText text-field "")))
       (.setPreferredSize (Dimension. 300 20)))
                
     (doto panel
       (.setPreferredSize (Dimension. 300 30))
       (.add prompt)
       (.add text-field)
       (.doLayout))

     (doto editor
       (.setContentType "text/clojure")
       (.setEditable false))
        
     (doto content
       (.setLayout (BorderLayout.))
       (.add scroll (BorderLayout/CENTER))
       (.add panel (BorderLayout/SOUTH)))

     (doto frame
       (.pack)        
       (.setVisible true)
       (.setDefaultCloseOperation JFrame/EXIT_ON_CLOSE))

     (doto text-field
       (.requestFocus)))))
