(ns overtone.at-at
  (:import [java.util.concurrent ScheduledThreadPoolExecutor TimeUnit]))

(defrecord PoolInfo [thread-pool jobs-ref id-count-ref])
(defrecord MutablePool [pool-ref stop-delayed? stop-periodic?])
(defrecord RecurringJob [id created-at ms-period initial-delay job pool-info description scheduled?])
(defrecord ScheduledJob [id created-at initial-delay job pool-info description scheduled?])

(defn- format-date
  "Format date object as a string such as: 15:23:35s"
  [date]
  (.format (java.text.SimpleDateFormat. "EEE hh':'mm':'ss's'") date))

(defmethod print-method PoolInfo
  [obj w]
  (.write w (str "#<PoolInfo: " (:thread-pool obj) " " (count @(:jobs-ref obj)) " jobs>")))

(defmethod print-method MutablePool
  [obj w]
  (.write w (str "#<MutablePool: " (count @(:jobs-ref @(:pool-ref obj))) " jobs, stop-delayed? " (:stop-delayed? obj) ", stop-periodic? " (:stop-periodic? obj) ">")))

(defmethod print-method RecurringJob
  [obj w]
  (.write w (str "#<RecurringJob id: " (:id obj) ", created-at: " (format-date (:created-at obj)) ", ms-period: " (:ms-period obj) ", initial-delay: " (:initial-delay obj) ", description: \"" (:description obj) "\", scheduled? " @(:scheduled? obj) ">")))

(defmethod print-method ScheduledJob
  [obj w]
  (.write w (str "#<ScheduledJob id: " (:id obj) ", created-at: " (format-date (:created-at obj)) ", initial-delay: " (:initial-delay obj) ", description: \"" (:description obj) "\", scheduled? " @(:scheduled? obj) ">")))

(defn- switch!
  "Sets the value of atom to new-val. Similar to reset! except returns the
  immediately previous value."
  [atom new-val]
  (let [old-val  @atom
        success? (compare-and-set! atom old-val new-val)]
    (if success?
      old-val
      (recur atom new-val))))

(defn- cpu-count
  "Returns the number of CPUs on this machine."
  []
  (.availableProcessors (Runtime/getRuntime)))

(defn- schedule-at-fixed-rate
  "Schedule the fun to execute periodically in pool-info's pool with the
  specified initial-delay and ms-period. Returns a RecurringJob record."
  [pool-info fun initial-delay ms-period description]
  (let [initial-delay (long initial-delay)
        ms-period     (long ms-period)
        t-pool        (:thread-pool pool-info)
        job           (.scheduleAtFixedRate t-pool
                                            fun
                                            initial-delay
                                            ms-period
                                            TimeUnit/MILLISECONDS)
        start-time    (System/currentTimeMillis)
        jobs-ref      (:jobs-ref pool-info)
        id-count-ref  (:id-count-ref pool-info)]
    (dosync
     (let [id       (commute id-count-ref inc)
           job-info (RecurringJob. id
                                   start-time
                                   ms-period
                                   initial-delay
                                   job
                                   pool-info
                                   description
                                   (atom true))]
       (commute jobs-ref assoc id job-info)
       job-info))))

(defn- wrap-fun-to-remove-itself
  [fun jobs-ref job-info-prom]
  (fn [& args]
    (let [job-info  @job-info-prom
          id        (:id job-info)
          sched-ref (:scheduled? job-info)]
      (reset! sched-ref false)
      (dosync
       (commute jobs-ref dissoc id))
      (apply fun args))))

(defn- schedule-at
  "Schedule the fun to execute once in the pool-info's pool after the specified
  initial-delay. Returns a ScheduledJob record."
  [pool-info fun initial-delay description]
  (let [initial-delay (long initial-delay)
        t-pool        (:thread-pool pool-info)
        jobs-ref      (:jobs-ref pool-info)
        id-prom       (promise)
        fun           (wrap-fun-to-remove-itself fun jobs-ref id-prom)
        job           (.schedule t-pool fun initial-delay TimeUnit/MILLISECONDS)
        start-time    (System/currentTimeMillis)
        id-count-ref  (:id-count-ref pool-info)
        job-info      (dosync
                       (let [id       (commute id-count-ref inc)
                             job-info (ScheduledJob. id
                                                     start-time
                                                     initial-delay
                                                     job
                                                     pool-info
                                                     description
                                                     (atom true))]
                         (commute jobs-ref assoc id job-info)
                         job-info))]
    (deliver id-prom job-info)
    job-info))

(defn- shutdown-pool-now!
  "Shut the pool down NOW!"
  [t-pool]
  (.shutdownNow t-pool))

(defn- shutdown-pool-gracefully!
  "Shut the pool down gracefully - waits until all previously submitted jobs
  have completed"
  [t-pool]
  (.shutdown t-pool))

(defn- mk-sched-thread-pool
  "Create a new scheduled thread pool containing num-threads threads."
  [num-threads stop-delayed? stop-periodic?]
  (let [t-pool (ScheduledThreadPoolExecutor. num-threads)]
    (doto t-pool
      (.setExecuteExistingDelayedTasksAfterShutdownPolicy (not stop-delayed?))
      (.setContinueExistingPeriodicTasksAfterShutdownPolicy (not stop-periodic?)))))

(defn- mk-pool-info
  [t-pool]
  (PoolInfo. t-pool (ref {}) (ref 0N)))

(defn mk-pool
  "Returns MutablePool record storing a mutable reference (atom) to a
  PoolInfo record which contains a newly created pool of threads to
  schedule new events for. Pool size defaults to the cpu count + 2. It
  is possible to modify the pool shutdown policy with the
  keys :stop-delayed? and :stop-periodic which indicates whether the
  pool should cancel all delayed or periodic jobs respectively on
  shutdown."
  [& {:keys [cpu-count stop-delayed? stop-periodic?]
      :or {cpu-count (+ 2 (cpu-count))
           stop-delayed? true
           stop-periodic? true}}]
  (MutablePool. (atom (mk-pool-info (mk-sched-thread-pool cpu-count stop-delayed? stop-periodic?)))
                stop-delayed?
                stop-periodic?))

(defonce default-pool* (mk-pool))

(defn every
  "Calls fun every ms-period, and takes an optional initial-delay for the first
  call in ms. Default pool is used if none explicity specified. Returns a
  scheduled-fn which may be cancelled with cancel

  Default options are
  {:initial-delay 0 :description \"\" :pool <default-pool>}"
  [ms-period fun & {:keys [initial-delay description pool]
                    :or {initial-delay 0
                         description ""
                         pool default-pool*}}]
  (schedule-at-fixed-rate @(:pool-ref pool) fun initial-delay ms-period description))

(defn now
  "Return the current time in ms"
  []
  (System/currentTimeMillis))

(defn at
  "Schedules fun to be executed at ms-time (in milliseconds). Executes
  immediately if ms-time is in the past. Default pool is used if none
  explicitly. Use (now) to get the current time in ms.

  Example usage:
  (at (+ 1000 (now))
      #(println \"hello from the past\")
      :description \"Message from the past\") ;=> prints 1s from now"
  [ms-time fun & {:keys [description pool]
                  :or {description ""
                       pool default-pool*}}]
  (let [initial-delay (- ms-time (now))
        pool-info  @(:pool-ref pool)]
    (schedule-at pool-info fun initial-delay description)))

(defn- shutdown-pool!
  [pool-info strategy]
  (let [t-pool (:thread-pool pool-info)]
    (case strategy
      :stop (shutdown-pool-gracefully! t-pool)
      :kill (shutdown-pool-now! t-pool))))

(defn stop-and-reset-pool!
  "Shuts down the threadpool of given MutablePool using the specified strategy
  (defaults to :stop). Shutdown happens asynchronously on a separate thread.
  The pool is reset to a fresh new pool preserving the original size. If called
  with no params, the default pool is used. Returns the old pool-info.

  Strategies for stopping the old pool:
  :stop - allows all running and scheduled tasks to complete before waiting
  :kill - forcefully interrupts all running tasks and does not wait

  Example usage:
  (stop-and-reset-pool!)                 ;=> default pool is reset gracefully
  (stop-and-reset-pool! :strategy :kill) ;=> default pool is reset forcefully
  (stop-and-reset-pool! :pool pool)      ;=> pool is reset gracefully
  (stop-and-reset-pool! :pool pool
                        :strategy :kill) ;=> pool is reset forcefully"
  [& {:keys [pool strategy]
      :or {pool default-pool*
           strategy :stop}}]
  (when-not (some #{strategy} #{:stop :kill})
    (throw (Exception. (str "Error: unknown pool stopping strategy: " strategy ". Expecting one of :stop or :kill"))))
  (let [pool-ref      (:pool-ref pool)
        num-threads   (.getCorePoolSize (:thread-pool @pool-ref))
        new-t-pool    (mk-sched-thread-pool num-threads (:stop-delayed? pool) (:stop-periodic? pool))
        new-pool-info (mk-pool-info new-t-pool)
        old-pool-info (switch! pool-ref new-pool-info)]
    (future (shutdown-pool! old-pool-info strategy))
    old-pool-info))

(defn- cancel-job
  "Cancel/stop scheduled fn if it hasn't already executed"
  [job-info cancel-immediately?]
  (if (:scheduled? job-info)
    (let [job       (:job job-info)
          id        (:id job-info)
          pool-info (:pool-info job-info)
          pool      (:thread-pool pool-info)
          jobs-ref  (:jobs-ref pool-info)]
      (.cancel job cancel-immediately?)
      (reset! (:scheduled? job-info) false)
      (dosync
       (let [job (get @jobs-ref id)]
         (commute jobs-ref dissoc id)
         (true? (and job (nil? (get @jobs-ref id))))))) ;;return true if success
    false))

(defn- cancel-job-id
  [id pool cancel-immediately?]
  (let [pool-info @(:pool-ref pool)
        jobs-info @(:jobs-ref pool-info)
        job-info (get jobs-info id)]
    (cancel-job job-info cancel-immediately?)))

(defn- job-kind
  [obj]
  (let [t (type obj)]
    (cond
     (= RecurringJob t) ::job-info
     (= ScheduledJob t) ::job-info
     (isa? t Number) ::job-id
     :else (throw (Exception. (str "Unknown job kind: " t))))))

(defn stop
  "Stop a recurring or scheduled job gracefully either using a
  corresponding record or unique id"
  [job-or-id & {:keys [pool]
                :or {pool default-pool*}}]
  (let [kind (job-kind job-or-id)]
    (case kind
      ::job-info (cancel-job job-or-id false)
      ::job-id   (cancel-job-id job-or-id pool false))))

(defn kill
  "kill a recurring or scheduled job forcefully either using a
  corresponding record or unique id"
  [job-or-id & {:keys [pool]
                :or {pool default-pool*}}]
  (let [kind (job-kind job-or-id)]
    (case kind
      ::job-info (cancel-job job-or-id true)
      ::job-id   (cancel-job-id job-or-id pool true))))

(defn scheduled-jobs
  "Returns a set of all current jobs (both scheduled and recurring) for the
  specified pool. Pool defaults to default-pool when not specified."
  ([] (scheduled-jobs default-pool*))
  ([mutable-pool]
     (let [pool-ref (:pool-ref mutable-pool)
           jobs     @(:jobs-ref @pool-ref)
           jobs     (vals jobs)]
       jobs)))

(defn- format-start-time
  [date]
  (if (< date (now))
    ""
    (str ", starts at: " (format-date date))))

(defn- recurring-job-string
  [job]
  (str "[" (:id job) "] [RECUR] created: "  (format-date (:created-at job)) (format-start-time (+ (:created-at job) (:initial-delay job))) ", period " (:ms-period job) "ms, desc: " (:description job)))

(defn- scheduled-job-string
  [job]
  (str "[" (:id job) "] [SCHED] created: " (format-date (:created-at job)) (format-start-time (+ (:created-at job) (:initial-delay job))) ", desc: " (:description job)))

(defn- job-string
  [job]
  (cond
   (= RecurringJob (type job)) (recurring-job-string job)
   (= ScheduledJob (type job)) (scheduled-job-string job)))

(defn print-schedule
  "Pretty print all scheduled jobs (in the default or specified pool)
  to stdout"
  ([] (print-schedule default-pool*))
  ([mutable-pool]
     (let [jobs (scheduled-jobs mutable-pool)]
       (if (empty? jobs)
         (println "No jobs are currently scheduled.")
         (dorun
          (map #(println (job-string %)) jobs))))))
