(ns leiningen.assemble
  (:require
   [aws.sdk.s3 :as s3]
   [clj-time.core :as time]
   [clj-time.format :as time-format]
   [clojure.core.async :as async]
   [clojure.java.io :as io]
   [clojure.pprint :refer [pprint]]
   [clojure.string :as string]
   [io.aviso.ansi :as ansi]
   [leiningen.core.classpath :as classpath]
   [leiningen.core.eval :refer [eval-in-project]]
   [leiningen.core.project :refer [merge-profiles]]
   [leiningen.help :as help]
   [me.raynes.conch :as conch]
   [me.raynes.conch.low-level :as sh]
   [me.raynes.fs :as fs]
   [net.cgrand.enlive-html :as enlive]
   [pandect.core :as digest]
   [pantomime.mime :refer [mime-type-of]]))

(defn message [& {:keys [title body tag-fn] :or {tag-fn identity}}]
  (let [tag (tag-fn (format "[%s] " title))
        replacement (str "$1" tag)
        body (string/replace body #"(^|\n)" replacement)]
    (println body)))

(defn info [s]
  (message
   :title "info"
   :body s
   :tag-fn ansi/bold-cyan))

(defn trim-slashes [s]
  (string/replace s #"^/+|/+$" ""))

(defn split-slashes [s]
  (string/split s #"/"))

(defn join-slashes [coll]
  (string/join "/" coll))

(defn ^Long timestamp []
  (time-format/unparse
   (time-format/formatters :basic-date-time)
   (time/now)))

(defn ^Boolean substring?
  "True if s2 is a substring of s1."
  [^String s1 ^String s2]
  (.contains s1 s2))

(defn ^String render-html
  "Given a sequence of Enlive HTML nodes, return a string of HTML."
  [enlive-nodes]
  (string/join (enlive/emit* enlive-nodes)))

(defn ^String timestamp-file-name
  "Given a file path return the file name with a timestamp."
  [^String file-path]
  (str (fs/base-name file-path true)
       "-"
       (timestamp)
       (fs/extension file-path)))

;; ---------------------------------------------------------------------
;; S3 asset management

(def ^:dynamic *aws-config*
  {:access-key nil
   :secret-key nil
   ;; Optional
   :token nil
   ;; S3 storage bucket.
   :s3-bucket nil
   ;; Root Cloudfront URL used in URL construction during the
   ;; compilation of HTML.
   :cloudfront-root-url nil
   ;; Path from the the root of the S3 bucket to store objects. Will
   ;; be appended to the value of :cloudfont-root-url during URL
   ;; construction and when uploading to S3.
   :s3-base-path nil})

(defmacro with-aws-config [config & body]
  `(binding [*aws-config* ~config] ~@body))

(defn ^String s3-bucket
  "Return the current value of :s3-bucket from *aws-config*."
  []
  (:s3-bucket *aws-config*))

(defn ^String s3-base-path
  "Return the current value of :s3-base-path from *aws-config*."
  []
  (:s3-base-path *aws-config*))

(defn ^String cloudfront-root-url
  "Return the current value of :cloudfront-root-url from *aws-config*."
  []
  (:cloudfront-root-url *aws-config*))

(defn aws-credentials
  "Return a map of the current AWS credentials from *aws-config*."
  []
  (select-keys *aws-config* [:access-key :secret-key :token]))

(defn cloudfront-url [& args]
  (->> (mapv trim-slashes args)
       (into [(cloudfront-root-url) (s3-base-path)])
       (join-slashes)))

(defn ^String s3-app-resource-path
  "Return a timestamped s3 resource path for an app-resource."
  [^String resource-path]
  (join-slashes ["app-resources" (timestamp-file-name resource-path)]))

(defn put-object-in-bucket [& args]
  (apply s3/put-object (aws-credentials) (s3-bucket) args))

(alter-meta! #'put-object-in-bucket assoc
             :arglists '([key value & [metadata & permissions]]))

(defn get-object-in-bucket [key]
  (s3/get-object (aws-credentials) (s3-bucket) key))

(defn object-exists-in-bucket? [key]
  (s3/object-exists? (aws-credentials) (s3-bucket) key))

(defn push-to-s3!
  "Push a file to s3 if it does not exist. Automatically determines
  the file content-type."
  [key file-path]
  (let [key (str (s3-base-path) "/" key)]
    (if (object-exists-in-bucket? key)
      (println (ansi/bold-yellow "[exists]") key)
      (let [content-type (mime-type-of file-path)
            file (io/file file-path)]
        (println (ansi/bold-green "[create]") key)
        (put-object-in-bucket key file {:content-type content-type})))))

;; TODO: Handle this in configuration. Use .gitignore.
(defn find-assets [folder]
  (let [pred (fn [file]
               (let [fullname (str file)
                     basename (fs/base-name fullname)]
                 (and (fs/file? file)
                      (not (or (= basename ".DS_Store")
                               (= basename "index.html")
                               ;; Vim/Emacs tmp files.
                               (re-matches #"(?:\#.*\#|.*\.swp)" basename)
                               ;; Ignore ClojureScript and Garden
                               ;; output directories.
                               (re-find #"/out/" fullname))))))]
    (fs/find-files* folder pred)))

;; SEE: push-all-resources-to-s3!
(def ^{:doc "Set of [s3-path path-on-disk] resources to push to s3."}
  s3-resources-to-push
  (atom []))

(defn add-s3-resource-to-push!
  [s3-path resource-path]
  (swap! s3-resources-to-push conj [s3-path resource-path]))

(defn add-assets-to-push!
  [folder]
  (let [re (re-pattern folder)]
    (doseq [file (find-assets folder)]
      (let [s3-path (->> (string/split (str file) re)
                         (second)
                         (trim-slashes))]
        (add-s3-resource-to-push! s3-path file)))))

(defn push-all-files-to-s3! []
  (doseq [[s3-path resource-path] @s3-resources-to-push]
    (push-to-s3! s3-path resource-path))
  (swap! s3-resources-to-push empty))

;; ---------------------------------------------------------------------
;; index.html transformation

(defn update-resource-paths
  "Prepend script.resource and link.resource tags' src and href
   with the cloudfront base URL."
  [nodes]
  (-> nodes
      (enlive/transform [:link.resource]
        (fn [node]
          (update-in node [:attrs :href] cloudfront-url)))
      (enlive/transform [:script.resource]
        (fn [node]
          (update-in node [:attrs :src] cloudfront-url)))))

(defn first-matching-build
  "Return the first build where the base name of 
  (get-in build [:compiler :output-to]) and path are equal."
  [builds ^String path]
  (some
   (fn [build]
     (let [output-to (get-in build [:compiler :output-to])]
       (when (= (fs/base-name output-to)
                (fs/base-name path))
         build)))
   builds))

(defn node->target-attr [node]
  (case (:tag node)
    :link :href
    :script :src))

(defn node->builds [node build-config]
  (case (:tag node)
    :link (:garden build-config)
    :script (:cljsbuild build-config)))

(defn ^String real-resource-path
  [node build-config]
  (let [attr (node->target-attr node)
        builds (node->builds node build-config)]
    (as-> (get-in node [:attrs attr]) ?x
          (first-matching-build builds ?x)
          (get-in ?x [:compiler :output-to]))))

;; TODO: If possible separate the concern of enqueing the resource
;; to push and the rewriting of the attribute value. One way we could
;; do this is to query for .app-resource hrefs and srcs, enque those,
;; then do the transformation associating the matched node with it's
;; new corresponding attribute value.
(defn transform-app-resource-node
  "Updates an app-resource node with a new cloudfront url."
  [node build-config]
  (let [attr (node->target-attr node)
        resource-path (real-resource-path node build-config)
        s3-path (s3-app-resource-path resource-path)]
    (add-s3-resource-to-push! s3-path resource-path)
    (assoc-in node [:attrs attr] (cloudfront-url s3-path))))

(defn transform-app-resource-nodes
  "Uploads .app-resources to s3 and changes the index file to have the
   s3 url."
  [nodes build-config]
  (enlive/transform nodes #{[:link.app-resource] [:script.app-resource]}
    (fn [node]
      (transform-app-resource-node node build-config))))

(defn transform-for-env
  "Remove nodes where the value of [data-env] is not equal to env."
  [nodes {:keys [env]}]
  (let [env (name env)]
    (enlive/transform nodes [:*] 
      (fn [node]
        (if-let [node-env (get-in node [:attrs :data-env])]
          (when (= node-env env)
            node)
          node)))))

(def
  ^{:doc "Matches an IE conditional comment. Captures the tag opener,
          content, and closer."}
  ie-comment-re
  #"(?s)(\[if(?: +(?:g|l)te?)? +IE +\d+\]\>)(.+)(<!\[endif\])")

(defn transform-ie-comments
  "Like update-resource-paths but applies only to script and link tags
   inside IE conditional comments."
  [nodes build-config]
  (let [f (fn [node]
            (if-let [[_ open content close] (re-matches ie-comment-re (:data node))]
              (let [new-content (-> (enlive/html-snippet content)
                                    (update-resource-paths)
                                    (transform-app-resource-nodes build-config)
                                    (transform-for-env build-config)
                                    (render-html))
                    new-comment (str open new-content close)]
                (update-in node [:data] (constantly new-comment)))
              node))]
    (enlive/transform nodes [enlive/comment-node] f)))

(defn transform-index-html
  [html-file-path build-config]
  (-> (enlive/html-resource html-file-path)
      (update-resource-paths)
      (transform-app-resource-nodes build-config)
      (transform-for-env build-config)
      (transform-ie-comments build-config)
      (render-html)))

;; XXX: Should this be configurable?
(defn ensure-dist-directory []
  (when-not (fs/directory? "dist")
    (fs/mkdir "dist")))

(defn build-index-html [build-config]
  (ensure-dist-directory)
  (->> (transform-index-html "resources/public/index.html" build-config)
       (spit "dist/index.html")))

;; ---------------------------------------------------------------------
;; CLI 

(def
  ^{:doc "A helpful dolphin to brighten even the darkest of terminals."}
  dolphin
  "
                                 _.-~~.
           _.--~~~~~---....__  .' . .,'
         ,'. . . . . . . . . .~- ._ (
        ( ..  ^  . . . . . . . . . .~-._
     .~__.-~    ~`. . . . . . . . . . . -.
     `----..._      ~-=~~-. . . . . . . . ~-.
               ~-._   `-._ ~=_~~--. . . . . .~.
                | .~-.._  ~--._-.    ~-. . . . ~-.
                 \\ .(   ~~--.._~'       `. . . . .~-.                ,
                  `._\\         ~~--.._    `. . . . . ~-.    .- .   ,'/
  _  . _ . -~\\        _ ..  _          ~~--.`_. . . . . ~-_     ,-','`  .
               ` ._           ~                ~--. . . . .~=.-'. /. `
         - . -~            -. _ . - ~ - _   - ~     ~--..__~ _,. /   \\  - ~
                . __ ..                   ~-               ~~_. (  `
  )`. _ _               `-       ..  - .    . - ~ ~ .    \\    ~-` ` `  `. _
        _ From Outpace with love _                - .  `  .   \\  \\ `.")

(defn read-lines<
  "Like sh/read-line but returns a channel ch containing each line from 
  from as it becomes available. ch closes when there are no more lines.
  Note, this should only be called once per process."
  [conch-process from]
  (let [ch (async/chan)]
    (async/go-loop []
      (if-let [line (sh/read-line conch-process from)]
        (do (async/>! ch line)
            (recur))
        (async/close! ch)))
    ch))

(defn run-lein
  "Run a leiningen command as a subprocess. Returns a future of the 
  process exit code.

  Options:
    :command (string) - the leiningen command to execute
    :title (string) - the tag for the output 

  Example:
    (run-lein {:command \"help\" :title \"[lein]\"})
  "
  [{:keys [command title]}]
  (message :title "exec" :body command :tag-fn ansi/bold-white)
  (let [args (string/split command #" +")
        proc (apply sh/proc "lein" args)
        exit (future (sh/exit-code proc))
        out-err-ch (async/merge [(read-lines< proc :out)
                                 (read-lines< proc :err)])]
    (async/go-loop []
      (when-let [line (async/<! out-err-ch)]
        (println title line)
        (recur)))

    exit))

(defn build-command
  "Helper function for building leinigen commands."
  [command-format & {:keys [title]}]
  (fn [watch? builds]
    (let [cmd (format command-format
                      (if watch? "auto" "once")
                      (string/join " " (map (comp name :id) builds)))]
      (run-lein {:command cmd :title title}))))

(def run-garden
  (build-command "garden %s %s"
                 :title (ansi/bold-green "[garden]")))

(def run-cljsbuild
  (build-command "do cljsbuild clean, cljsbuild %s %s"
                 :title (ansi/bold-blue "[cljsbuild]")))

(defn build
  [{auto :auto? garden-builds :garden cljs-builds :cljsbuild}]
  (println (ansi/bold-blue dolphin))
  (let [garden-exit (run-garden auto garden-builds)
        cljsbuild-exit (run-cljsbuild auto cljs-builds)
        exit-sum (+ (Math/abs @garden-exit)
                    (Math/abs @cljsbuild-exit))]
    ;; Sum the exit codes. Non-zero means something failed.
    (when (< 0 exit-sum)
      (info "Failure")
      (System/exit exit-sum))))

(defn build-release
  "Push resources to S3 and compile production JavaScript, CSS, and
  HTML."
  [build-config]
  (build build-config)

  (info "Compiling index.html...")
  (build-index-html (assoc build-config :env :prod))

  (info "Pushing assets to S3...")
  (add-assets-to-push! "resources/public")  
  (push-all-files-to-s3!))

;; ---------------------------------------------------------------------
;; Project utilities

(defn get-aws-config
  "Extract assemble AWS configuration from a projec."
  [project]
  (get-in project [:outpace/assemble :aws-config]))

(defn get-build
  "Returns the first matching build (a map) with build-id in a 
  collection of builds.

  Ex.
    (let [builds (get-in project [:cljsbuild :builds])]
      (get-build builds \"dev\"))
    ;; => {:id \"dev\", :compiler {...}}
  "
  [builds build-id]
  (as-> (group-by :id builds) ?x
        (get ?x build-id)
        (first ?x)))

(defn get-build-opts
  "Find the first Assemble build with build-id."
  [project build-id]
  (-> (get-in project [:outpace/assemble :builds])
      (get-build build-id)))

(defn assemble-build-config
  "Given a project, build-type, and build-id, return a build 
  configuration map containing all relevant data for build processes."
  [project build-type build-id]
  (when-let [build-opts (get-build-opts project build-id)]
    (letfn [(ids->builds [m k]
              (update-in m [k]
                         (fn [build-ids]
                           (->> (mapv (fn [id]
                                        (-> (get-in project [k :builds])
                                            (get-build id)))
                                      build-ids)
                                (filter boolean)))))]
      (-> build-opts 
          (assoc :build-name (:name project)
                 :auto? (= build-type "auto"))
          (ids->builds :garden)
          (ids->builds :cljsbuild)))))

(defn validate-args!
  "Aborts when the build-type and build-id parameters are not passed to 
  the assemble command."
  [[build-type build-id]]
  (when-not (and build-type build-id)
    (println "ERROR: Both a build-type and build-id must provided")
    (System/exit 1)))

(defn invalid-aws-config-keys
  "Given a map containing AWS configuration return a seq of keys for 
  which the value is not a string."
  [aws-config]
  (for [k [:access-key :secret-key]
        :let [v (get aws-config k)]
        :when (not (string? v))]
    k))

(defn validate-aws-config!
  "Aborts when :access-key and :secret-key are invalid."
  [aws-config]
  (when-let [invalid-keys (seq (invalid-aws-config-keys aws-config))]
    (let [msg-fmt "ERROR: Invalid or missing AWS configuration %s."
          msg (->> (map pr-str invalid-keys)
                   (string/join ", ")
                   (format msg-fmt))]
      (println msg)
      (System/exit 1))))

(defn assemble
  "Orlando Bloom builds your ClojureScript project!"
  [project & [build-type build-id :as args]]
  (validate-args! args)
  (if-let [build-config (assemble-build-config project build-type build-id)]
    (let [aws-config (get-aws-config project)]
      (if (= build-type "release")
        (do
          (assert aws-config "Missing configuration")
          (validate-aws-config! aws-config)
          (with-aws-config aws-config
            (build-release build-config)))
        (build build-config)))
    (do (println "Could not locate build" build-id)
        (System/exit 1))))
