(ns cutty-sark.logging
  "This is drop-in replacement for io.pedestal.log.
  Wraps pedestal logging, serialising to json instead of edn
  and providing request correlation for better visibility"
  (:require [io.pedestal.log :as log]
            [cheshire.core :as json]
            [cutty-sark.correlation-ctx :as correlation-ctx]))


(defn json-formatter
  "Tries to serialize log arguments to json.
  If it fails, then produces a json with raw key and value generated by pr-str"
  [args-map]
  (try (json/generate-string args-map)
       (catch Exception _ (json/generate-string {:raw (pr-str args-map)}))))


(defn replace-formatter
  "Use this in your local development setup to produce more readable logs locally"
  ([] (replace-formatter pr-str))
  ([formatter] (alter-var-root #'json-formatter (constantly formatter))))


(def context-filter identity)


(defn set-context-filter! [ctx->ctx]
  (alter-var-root #'context-filter (constantly ctx->ctx)))


(defn- log
  "Formats logs, providing extra information:
  - line number
  - flow-id to correlate requests"
  [form level keyvals]
  (let [keyvals-map (apply array-map keyvals)]
    `(log/log (into ~keyvals-map
                    {::log/formatter   json-formatter
                     ::log/logger-name ~(name (ns-name *ns*))
                     :line             (:line ~(meta form))
                     :context          (cond-> correlation-ctx/*ctx*
                                               (not (= ::correlation-ctx/global correlation-ctx/*ctx*))
                                               (context-filter))})
              ~level)))


(defmacro trace [& keyvals] (log &form :trace keyvals))

(defmacro debug [& keyvals] (log &form :debug keyvals))

(defmacro info [& keyvals] (log &form :info keyvals))

(defmacro warn [& keyvals] (log &form :warn keyvals))

(defmacro error [& keyvals] (log &form :error keyvals))

(defmacro spy
  "Logs expr and its value at DEBUG level, returns value."
  [expr]
  (let [value# (gensym "value")]
    `(let [~value# ~expr]
       ~(log &form :debug (vector :spy (pr-str expr)
                                  :value value#))
       ~value#)))

(comment

  (macroexpand-1 '(debug :msg "hehe"))

  (log/debug :msg "hehe")

  (spy (+ 1 1))

  (defn pprint-formatter
    "Pretty-prints arguments map"
    [args-map]
    (with-out-str
      (println)
      (clojure.pprint/pprint args-map)))

  )
