(ns clj-opentracing.explicit-tracer
  (:require [clojure.tools.logging :as log]
            [clojure.string :as str])
  (:import (io.opentracing.propagation TextMapInjectAdapter Format$Builtin TextMapExtractAdapter)
           (java.util HashMap)))


(def ^:dynamic *current-span* nil)


(defmacro ^:private with-assert-inside-span [tracer & body]
  `(when ~tracer
     (assert *current-span* "Must be called from inside a span.")
     (when *current-span*
       ~@body)))


(defn set-tag! [tracer k v]
  (with-assert-inside-span tracer
    (.setTag *current-span* (name k) (str v))))


(defn set-tags! [tracer tag-map]
  (doseq [[k v] tag-map]
    (set-tag! tracer k v)))


(defn normalize-string-or-map [string-or-map]
  (if (map? string-or-map)
    (into {} (for [[k v] string-or-map
                   :when (some? v)]
               [(name k) v]))
    (if (nil? string-or-map)
      ""
      string-or-map)))


(defn log!
  ([tracer string-or-map]
   (with-assert-inside-span tracer
     (.log *current-span* (normalize-string-or-map string-or-map))))
  ([tracer timestamp string-or-map]
   (with-assert-inside-span tracer
     (.log *current-span* timestamp (normalize-string-or-map string-or-map)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Creating spans


(defn with-span-impl
  [tracer span-name span-or-context f]
  ;; We don't rely on io.opentracing.ScopeManager to get current span because its thread-local value is not propagated
  ;; through pmap, future, send, send-off, core.async/thread and other clojure ways to create a new thread
  ;; However, we still create a Scope for each new span so that it automatically gets closed
  ;; ACHTUNG we don't know what happens if the parent span is closed before its child is closed
  (if-not tracer
    (f)
    (with-open [scope (-> tracer
                          (.buildSpan span-name)
                          (.asChildOf span-or-context)      ; nil-friendly
                          (.startActive true))]
      (binding [*current-span* (.span scope)]
        (try
          (f)
          (catch Exception e
            (set-tag! tracer :error true)
            (log! tracer {:error.kind   "Exception"
                          :error.object e})
            (throw e)))))))


(defmacro with-span [tracer span-name & body]
  `(with-span-impl ~tracer ~span-name *current-span* (fn [] ~@body)))


;; Explicitly setting parent span
(defmacro with-child-span [tracer span-name parent-span-or-context & body]
  `(with-span-impl ~tracer ~span-name ~parent-span-or-context (fn [] ~@body)))


;; wrap-span is a Ring middleware, first argument must be handler
(defn wrap-span [handler tracer span-name tags]
  (fn [request]
    (with-span tracer span-name
      (set-tags! tracer tags)
      (handler request))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Using parent spans created by our server's clients


(defn extract-span-context [tracer payload format]
  (when tracer
    (try
      (let [extract-adapter (TextMapExtractAdapter. payload)]
        (.extract tracer format extract-adapter))
      (catch Exception e
        (log/warn e "Unable to parse parent span headers.")))))

(defn headers->span-context [tracer headers]
  (extract-span-context tracer headers Format$Builtin/HTTP_HEADERS))

(defn text-map->span-context [tracer text-map]
  (extract-span-context tracer text-map Format$Builtin/TEXT_MAP))

;; wrap-span-from-request is a Ring middleware, first argument must be handler
;; Additionally sets server tags (for request and response)
(defn wrap-span-from-request [handler tracer span-name additional-tags]
  (fn [{:as request :keys [headers uri request-method]}]
    (with-child-span tracer span-name (headers->span-context tracer headers)
      (set-tags! tracer (merge {:span.kind   "server"
                                :http.url    (or uri "")
                                :http.method (str/upper-case (name (or request-method "")))
                                :user-agent  (get headers "user-agent" "")}
                               additional-tags))
      (let [res (handler request)]
        (set-tags! tracer {:http.status_code (:status res)})
        res))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; For us as a client to include our current span context in a HTTP request to another service


(defn get-span-context [tracer span format]
  (when tracer
    (when-let [current-span-context (some-> span .context)]
      (let [mutable-carrier-map (HashMap.)
            inject-adapter      (TextMapInjectAdapter. mutable-carrier-map)]
        (.inject tracer current-span-context format inject-adapter)
        (into {} mutable-carrier-map)))))


(defn get-current-span-context-headers [tracer]
  (get-span-context tracer *current-span* Format$Builtin/HTTP_HEADERS))

(defn get-current-span-context-text-map [tracer]
  (get-span-context tracer *current-span* Format$Builtin/TEXT_MAP))
