;;;; Copyright (c) 2015 William Yao

;;;; This software is provided with no license whatsoever. Redistributions
;;;; or use in source or binary form are permitted with no restrictions.
;;;; Acknowledgement would be nice, but is not required.

;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR 'AS IS' AND ANY EXPRESSED
;;; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;;; keyword multimethods, allowing different methods to have different
;;;; keyword arguments

(ns leiningen.keyfn.keyfn)

;;;; basically just need to append a new argument onto the end of the
;;;; argument list when defining a multimethod, which will be a hash table
;;;; and then generate some other stuff... all the actual multimethods
;;;; will be shadowed and a normal function will generate the hashes.

;;; the symbol used to specify the start of keyword args
(declare $)

(def
  ^{:doc "The suffix added to the name of a defined keyfn to create
         the shadowed methods."} 
  keyfn-suff 
  "-")

(defn sym
  "Create a new symbol in package from concatenation of arguments, as per STR"
  [& args]
  (symbol (reduce str args)))

(defn keyfn-name [name]
  (sym name keyfn-suff))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn plmap
  "PLIST to MAP"
  [plist]
  (into {} (map vec (partition 2 plist))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro with-keys
  "Evaluate BODY in scope from pulling values from HASH using the 
  specifications from KEYS.
  
  Each entry is KEYS is either a symbol, where a keyword from the symbol
  will be used from the hash, or a sequence of
  (name default? provided?)
  where name can either be a symbol or
  (key name), where key is the key used to pull values from the hash and
  name is the variable to bind to."
  [keys hash & body]
  (letfn [(norm-spec [spec]
            (if (seq? spec)
              (let [name (first spec)
                    norm-name (if (seq? name)
                                name
                                [(keyword name) name])]
                (condp = (count spec)
                       3 `[~norm-name ~@(next spec)]
                       2 `[~norm-name ~@(next spec) nil]
                       1 `[~norm-name nil nil]
                       (throw (Exception. "Malformed key specification."))))
              [[(keyword spec) spec] nil nil]))]
    `(let [~@(loop [binds []
                    keys (map norm-spec keys)]
                   (if (empty? keys)
                     binds
                     (let [[[key name] default? provided?] (first keys)
                           
                           binds (conj binds
                                       name
                                       `(get ~hash ~key ~default?))
                           binds (if provided?
                                   (conj binds
                                         provided?
                                         `(contains? ~hash ~key))
                                   binds)]
                       (recur binds
                              (rest keys)))))]
       ~@body)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro defkeyfn
  "Define a new keyfn.
  The start of the keys is specified by $; incompatible with variadic args."
  {:arglists '([name doc-string? attr-map? [& params $ keys] prepost-map? & body])}
  [name & options]
  (let [doc-string       (and (string? (first options))
                              (first options))
        options          (if (string? (first options))
                           (next options)
                           options)
        attr-map         (if (map? (first options))
                           (first options)
                           {})
        options          (if (map? (first options))
                           (next options)
                           options)
        [[& formal-args],
         [_ & key-args]] (if (vector? (first options))
                           (split-with (complement #{(symbol '$)})
                                       (first options))
                           (throw (Exception.
                                    "ERROR: argument list must be a vector")))
        options          (next options)
        prepost-map      (if (map? (first options)) 
                           (first options)
                           {})
        body             (if (map? (first options))
                           (next options)
                           options)
        attr-map         (if doc-string
                           (assoc attr-map :doc doc-string)
                           attr-map)
        attr-map         (assoc attr-map 
                                :arglists
                                `'~`([~@formal-args ~@(when key-args
                                                        `($ ~@key-args))]))]
    `(defn ~name ~attr-map
       [~@formal-args & kvs#]
       ~prepost-map
       (with-keys ~key-args (plmap kvs#)
         ~@body))))

(defn signature-code
  {:arglists '([[[& params] prepost-map? & body]])}
  [[[& params] & options]]
  (let [prepost-map (if (map? (first options))
                      (first options)
                      {})
        body        (if (map? (first options))
                      (next options)
                      options)

        [[& formal-args] [_ & key-args]] (split-with (complement #{(symbol '$)})
                                                     params)]
    `([~@formal-args hash#]
      ~prepost-map
      (with-keys ~key-args hash#
        ~@body))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro defkeymulti
  "Define a new keymulti; methods on this function that take differentiated
  keyword arguments are then defined using DEFKEYMETHOD.
  
  Multimethods defined using DEFKEYMULTI must not specify variadic arguments."
  {:arglists '([name doc-string? attr-map? dispatch [& args] & options])}
  [name & options]
  (let [doc-string (and (string? (first options))
                        (first options))
        options    (if (string? (first options))
                     (next options)
                     options)
        attr-map   (if (map? (first options))
                     (first options)
                     {})
        options    (if (map? (first options))
                     (next options)
                     options)
        dispatch   (first options)
        [& args]   (second options)
        options    (nnext options)
        attr-map   (if doc-string
                     (assoc attr-map :doc doc-string)
                     attr-map)

        multi-name (keyfn-name name)]
    `(do
       (defn ^{:doc ~doc-string} ~name
         [~@args & key-value]
         (~multi-name ~@args (plmap key-value)))
       (defmulti ~attr-map ~multi-name
         (fn [~@args _] ~dispatch)
         ~options))))

(defmacro defkeymethod
  "Define a new keymethod on specified keymulti, with optional key arguments.
  The start of the keys is specified by $.
  
  Each keyword argument can take the forms:
  name           ==> keyword will be constructed from name, no default value
  (name default) ==> keyword will be constructed from name, default = default"
  {:arglists '([name dispatch-val [& args $ keys] & body])}
  [name dispatch-val [& args] & body]
  (letfn [(normalize-arg [arg]
            (if (seq? arg)
              (condp = (count arg)
                     2 arg
                     (throw (Exception. "Incorrect argument form.")))
              (list arg nil)))]
    `(defmethod ~name ~dispatch-val
       ~@(signature-code `(~args ~@body)))))
