(ns clinicaltrials-gov.data
  (:require [clojure.data.zip :as z]
            [clojure.data.zip.xml :as x]
            [clojure.string :refer [trim]]))

(comment
  The macro below is a convenience function to make it more concise to define
  xml traversal functions.

  Each "row" creates a different function.

  [nct-id :id_info :nct_id x/text] would expand into the following...

  (defn nct-id
    [z]
    (x/xml1-> z
              :id_info
              :nct_id
              x/text)))

(defmacro create-fns
  [& body]
  (cons `do
        (for [b body]
          `(defn ~(first b) [z#]
             (x/xml1-> z# ~@(rest b))))))

(create-fns
  [nct-id :id_info :nct_id x/text]
  [org-study-id :id_info :org_study_id x/text]
  [brief-title :brief_title x/text]
  [official-title :official_title x/text]
  [lead-sponsor :sponsors :lead_sponsor :agency x/text]
  [brief-summary :brief_summary :textblock x/text trim]
  [overall-status :overall_status x/text]
  [phase :phase x/text]
  [study-type :study_type x/text]
  [study-design :study_design x/text trim]
  [condition :contition x/text]
  [eligibility-text :eligibility :criteria :textblock x/text trim]
  [eligibility-gender :eligibility :gender x/text]
  [eligibility-min-age :eligibility :minimum_age x/text]
  [eligibility-max-age :eligibility :maximum_age x/text]
  [eligibility-healthy-volunteers? :eligibility :healthy_volunteers x/text])

(defmacro create-list-fns
  "`create-fns` uses x/xml1-> while this uses x/xml->"
  [& body]
  (cons `do
        (for [b body]
          `(defn ~(first b) [z#]
             (x/xml-> z# ~@(rest b))))))

(create-list-fns
  [intervention-type :intervention :intervention_type x/text]
  [intervention-name :intervention :intervention_name x/text]
  [intervention-desc :intervention :description x/text]
  [intervention-arm-group-label :intervention :arm_group_label x/text]
  [intervention-other-name :intervention :other_name x/text]
  [facility-names :location :facility :name x/text]
  [facility-country :location :facility :address :country x/text]
  [facility-zip :location :facility :address :zip x/text]
  )

(defn keywords
  [z]
  (x/xml-> z :keyword x/text))