(ns open-scad.watch
  (:require [clojure.set                        :as set]
            [clojure.tools.namespace.track      :as track]
            [clojure.tools.namespace.dependency :as deps]
            [clojure.tools.namespace.dir        :as dir]
            [clojure.tools.namespace.file       :as file]
            [clojure.tools.namespace.find       :as find]
            [clojure.tools.namespace.parse      :as parse]
            [clojure.tools.namespace.reload     :as load ]
            [clojure.tools.namespace.repl       :as repl]
            [hawk.core                          :as hawk]
            [clojure.java.classpath             :refer [classpath-directories]]
            [clojure.pprint                     :refer [pprint]]
            [shuriken.namespace                 :refer [with-ns]]
            [threading.core :refer :all]))

(with-ns 'user
  (require '[clojure.tools.namespace.repl :refer [refresh]])
  (require '[clojure.pprint :refer [pprint]]))

(repl/disable-reload!)

;; TODO: move to shuriken.term
(defn clear []
  ; clear screen
  (print (str (char 27) "[2J"))
  ; move cursor to the top left corner of the screen
  (print (str (char 27) "[;H")))

(defonce refresh-tracker (track/tracker))
(defonce boot-time       (new java.util.Date))

(def referred               #'repl/referred)
(def aliased                #'repl/aliased)
(def remove-disabled        #'repl/remove-disabled)
(def print-pending-reloads  #'repl/print-pending-reloads)
(def print-and-return       #'repl/print-and-return)
(def recover-ns             #'repl/recover-ns)

(def find-files             #'dir/find-files)

(def watched-dirs (classpath-directories))

;; Adapted from https://github.com/clojure/tools.namespace/blob/0ce6646bb59a319a864faad5d01f1bc31c79e932/src/main/clojure/clojure/tools/namespace/repl.clj#L83-L109
(defn- refresh-recently-edited-files []
  (let [current-ns-name    (ns-name *ns*)
        current-ns-refers  (referred *ns*)
        current-ns-aliases (aliased *ns*)
        tracker            (dir/scan-files
                             (track/tracker)
                             (find-files watched-dirs find/clj))
        deps               (-> tracker ::track/deps deps/map->MapDependencyGraph)
        files              (->> (find-files watched-dirs find/clj)
                                (filter #(-> % .lastModified (java.util.Date.)
                                             (.after boot-time)))
                                set)
        file->ns-name      (::file/filemap tracker)
        ns-name->file      (set/map-invert file->ns-name)
        files-to-reload    (fn [f]
                             (->> (map file->ns-name files)
                                  (f deps)
                                  (map ns-name->file)
                                  (filter (::dir/files tracker))
                                  set
                                  (set/union files)))]
    (alter-var-root
      #'refresh-tracker dir/scan-files
      (set/union (files-to-reload deps/transitive-dependencies-set)
                 (files-to-reload deps/transitive-dependents-set))
      {:platform find/clj})
    (alter-var-root #'refresh-tracker remove-disabled)
    (print-pending-reloads refresh-tracker)
    (alter-var-root #'refresh-tracker load/track-reload)
    ; (in-ns current-ns-name)
    (let [result (print-and-return refresh-tracker)]
      (if (= :ok result)
        result
        ;; There was an error, recover as much as we can:
        (do (when-not
              (or (false? (:clojure.tools.namespace.repl/unload (meta *ns*)))
                  (false? (:clojure.tools.namespace.repl/load   (meta *ns*))))
              (recover-ns current-ns-refers current-ns-aliases))
            ;; Return the Exception to the REPL:
            result)))))

(hawk/watch!
  [{:paths watched-dirs
    :handler (fn [_ctx _opts]
               (clear)
               (pprint (refresh-recently-edited-files))
               (newline))}])
