(ns open-scad.models.led-holder-2
  (:require [open-scad.core :refer :all]
            [open-scad.libs.dot-scad :refer [rounded-cube]]))

(def bar-diameter               10)
(def roundness                  1)

(def holder-width               20)
(def holder-length              45)
(def holder-height              13)

(def terminus-width             5)
(def terminus-length            holder-length)
(def terminus-height            holder-height)


(def screw-hole-diameter         3.2)
(def screw-height                8)
(def screw-safety-height         1)

(def hanging-ring-inner-diameter 8)
(def hanging-ring-thickness      2)

(def washer-diameter             15)
(def washer-height               1.3)

(defgeometry holes []
  (let [r      (/ bar-diameter 2)
        hole   (->> (cylinder r (+ holder-width 2))
                    (rotate [0 (° 90) 0])
                    (translate [(/ holder-width 2)
                                (/ holder-length 2)
                                (/ holder-height 2)]))
        offset (+ (/ r 2) (/ holder-length 4))
        hs     (union (translate [0 (+ offset) 0] hole)
                      (translate [0 (- offset) 0] hole))]
    (union hs (translate [0 0 r] hs))))

(defgeometry screw-hole [h]
  (cylinder (/ screw-hole-diameter 2) h))

(defgeometry led-holder []
  (let [h (+ screw-height screw-safety-height 1)]
    (-> (rounded-cube [holder-width holder-length holder-height] roundness)
        (difference (holes)
                    (->> (screw-hole h)
                         (translate [(/ holder-width 2)
                                     (/ holder-length 2)
                                     (- holder-height (dec (/ h 2)))]))))))

(defgeometry hanging-ring []
  (let [radius (+ (/ hanging-ring-inner-diameter 2) hanging-ring-thickness)]
    (->> (extrude-rotate (translate [(/ hanging-ring-inner-diameter 2) 0 0]
                                   (circle hanging-ring-thickness)))
         (rotate [0 (° 90) 0])
         (translate
           [(/ terminus-width 2) (/ holder-length 2) (+ hanging-ring-thickness
                                                        holder-height)]))))

(defgeometry terminus []
  (-> (rounded-cube [terminus-width terminus-length terminus-height] roundness)
      (difference (holes))
      (union (hanging-ring))))

(defgeometry washer []
  (-> (cylinder (/ washer-diameter 2) washer-height)
      (difference (screw-hole (+ 2 washer-height)))))


(render ($fn 40 (union (part :holder    (led-holder))
                       (->> (part :terminus (terminus))
                            (translate [-20 0 0]))
                       (->> (part :washer (washer))
                            (translate [0 -20 0])))))
