(ns open-scad.libs.clip
  (:require [threading.core :refer :all]
            [weaving.core :refer :all]
            [open-scad.core :refer :all]
            [open-scad.libs.bosl.transforms :refer [rot]]))

(def ^:dynamic *clip-opening-ratio*       75/100)
(def ^:dynamic *clip-thickness-ratio*     1/4)
(def ^:dynamic *clip-screw-support-ratio* 1/2)
(def ^:dynamic *clip-screw-support-angle* 12)
(def ^:dynamic *bolt-height*              2.35)
(def ^:dynamic *bolt-out-diameter*        (+ 6.00 1))
(def ^:dynamic *bolt-in-diameter*         (+ 2.90 0.3))

(defgeometry clip-tip [clip]
  (let [self         (clip-tip clip)
        op           (opening clip)
        thick        (thickness self)
        screw-hole   (->> (cylinder
                            (* 1.05 (/ *bolt-in-diameter* 2))
                            (+ op (* 2 thick) 2))
                          (rotate [(° 90) 0 0]))]
    (difference (cube (length self) thick (height self))
                screw-hole)))

(defmethod length ::clip-tip [[_type {:keys [clip]}]]
  (max (+ (thickness clip)
          (* *clip-screw-support-ratio* (width clip)))
       (* 1.5 *bolt-out-diameter*)))
(defmethod height ::clip-tip [[_type {:keys [clip]}]]
  (max (height clip) (* 1.1 *bolt-out-diameter*)))
(defmethod thickness ::clip-tip [[_type {:keys [clip]}]]
  (max (thickness clip) *bolt-height*))

(defgeometry add-screw-tips [clip]
  (let [op         (opening clip)
        op-offset  (√ (- (** (radius clip)) (** (/ op 2))))
        proto-tip  (clip-tip clip)
        proto-l    (length proto-tip)
        thick      (thickness proto-tip)
        bolt       (with-fn 6
                     (->> (cylinder (/ *bolt-out-diameter* 2) thick)
                          (rotate [(° 90) 0 0])
                          (translate [0  (/ thick 2) 0])))
        x          (+ (/ proto-l 2) op-offset)
        move-left  (|| translate [x  (/ (+ op thick) 2)      0])
        move-right (|| translate [x  (- (/ (+ op thick) 2))  0])
        rotate-tip #(rot [0 0 %1] {:cp %2} %3)
        rot-left   (|| rotate-tip *clip-screw-support-angle*
                       [(/ op 2)  (/ op 2)                 0])
        rot-right  (|| rotate-tip (- *clip-screw-support-angle*)
                       [(/ op 2)  (- (+ (/ op 2)  thick))  0])
        ltip       (move-left (difference proto-tip bolt))
        rtip       (move-right proto-tip)
        proto-neg  (scale [1 1 1.1] proto-tip)
        negltip    (translate [0 (- thick) 0] (move-left  proto-neg))
        negrtip    (translate [0 thick     0] (move-right proto-neg))]
    (difference (union clip
                       (rot-left  ltip)
                       (rot-right rtip))
                (rot-left  negltip)
                (rot-right negrtip))))

(defmethod width     ::add-screw-tips [[_ {:keys [clip]}]]  (width     clip))
(defmethod height    ::add-screw-tips [[_ {:keys [clip]}]]  (height    clip))
(defmethod radius    ::add-screw-tips [[_ {:keys [clip]}]]  (radius    clip))
(defmethod thickness ::add-screw-tips [[_ {:keys [clip]}]]  (thickness clip))
(defmethod opening   ::add-screw-tips [[_ {:keys [clip]}]]  (opening   clip))

(defgeometry clip* [w h]
  (let [h         (max h (* 1.1 *bolt-out-diameter*))
        self      (clip* w h)
        radius    (radius self)
        cy-radius (max (* (+ 1 *clip-thickness-ratio*) radius)
                       (* (+ 1 (/ *bolt-height* radius)) radius))
        out-cy    (cylinder cy-radius h)
        in-cy     (cylinder radius (+ 2 h))
        opening   (* *clip-opening-ratio* w)
        op-offset (√ (- (** radius) (** (/ opening 2))))]
    (difference out-cy
                in-cy
                (translate [(+ (/ opening 2) op-offset) 0 0]
                           (cube opening opening (+ 2 h))))))

(defmethod width     ::clip* [[_ {:keys [w]} :as x]] (+ w (thickness x)))
(defmethod height    ::clip* [[_ {:keys [h]}]] h)
(defmethod radius    ::clip* [[_ {:keys [w]}]] (/ w 2))
(defmethod opening   ::clip* [[_ {:keys [w]} :as x]] (* *clip-opening-ratio* w))
(defmethod thickness ::clip* [[_ {:keys [w]} :as x]] (* *clip-thickness-ratio* (/ w 2)))

(defn clip [w h & {:keys [screw]}]
  (-> (clip* w h)
      (when-> (<- screw) add-screw-tips)))
