(ns open-scad.models.propagation-box
  (:require [open-scad.core :refer :all]
            [threading.core :refer :all]))

(def tube-diameter         9)
(def bottom-segment-length 100)
(def bottom-segment-height 1)
(def clip-thickness        1.5)
(def clip-length           8)

;; (union (->> (cube (* (- 2.5 1/2) tube-diameter)
;;                   (* 1.5 tube-diameter)
;;                   bottom-segment-height)
;;             (translate [(- (xsign (* 2.5 tube-diameter)))
;;                         0
;;                         (* -1/2 bottom-segment-height)])))

(defgeometry clip [& {:keys [middle]}]
  (let [cyl (cylinder (/ tube-diameter 2) (+ 1 (* 1.5 tube-diameter)))]
    (-> (->> (cube (* (- 2.5 1/4) tube-diameter)
                   (* 1.5 tube-diameter)
                   (+ (* 3/4 tube-diameter) bottom-segment-height))
             (translate [(* 1/8 tube-diameter) 0 (- (* -1/8 tube-diameter)
                                                    (* 1/2 bottom-segment-height))]))
        (difference (->> cyl
                         (translate [(* -3/4 tube-diameter) 0 0]))
                    (->> cyl
                         (rotate [(° 90) 0 0])
                         (translate [(* 2/4 tube-diameter) 0 0])))
        (when-not-> (<- middle)
                (union (->> (cube (* (- 2.5 1/4) tube-diameter)
                                  (* 1.5 tube-diameter)
                                  (+  bottom-segment-height))
                            (translate [(* 1/8 tube-diameter)
                                        0
                                        (+ (* -1/2 (+ bottom-segment-height tube-diameter)))]))))
        (->> (translate [0 0 (* 1/2 tube-diameter)])))))

(defgeometry full-plate []
  (cube bottom-segment-length bottom-segment-length bottom-segment-height))

(defgeometry box-bottom [& [plate & {:keys [middle]}]]
  (union (or plate (full-plate))
         (for [xsign [+ -]
               ysign [+ -]
               :let  [offset (/ bottom-segment-length 2)]]
           (->> (clip :middle middle)
                (>>- (when-> (<- (= xsign +))
                             (->> (rotate [0 0 (° 180)]))))
                (translate [(xsign (- offset (* 1/4 tube-diameter)))
                           (ysign (- offset (* 3/4 tube-diameter)))
                           (* 1/2 bottom-segment-height)])))))

(defgeometry grid-plate [n]
  (-> (union (let [t 2
                   w (+ (/ bottom-segment-length n) t)
                   g (->> (for [i (range (/ (/ n 1) 2))]
                            (->> (cube t bottom-segment-length bottom-segment-height)
                                 (translate [(+ (* 1/2 t) (* i w)) 0 0])))
                          (translate [(* -1/2 bottom-segment-length) 0 0]))
                   h (union g (->> g (rotate [0 0 (° 180)])))]
               (union h (->> h (rotate [0 0 (° 90)])))))))

(defgeometry box-top []
  (box-bottom (grid-plate 5)))

(defgeometry box-middle []
  (box-bottom (grid-plate 7) :middle true))


(render ($fn 100 [(->> (part :top    (box-top)))
                  (->> (part :middle (box-middle))
                       (translate [(* 3 bottom-segment-length)   0 0]))
                  (->> (part :bottom (box-bottom))
                       (translate [(* 1.5 bottom-segment-length) 0 0]))]))
