(ns open-scad.models.kossel-double-effector
  (:require [open-scad.core :refer :all :exclude [cylinder]]
            [open-scad.libs.kossel :as kossel]
            [open-scad.libs.mcad.stepper :refer [motor]]
            [open-scad.models.extruder-adapter :refer [extruder-adapter
                                                       adapter-height]]
            [threading.core :refer :all]))

(defgeometry cylinder [& {:keys [h r r1 r2 d d1 d2 center $fa $fs $fn]}])

(def effector-height           8)
(def hole-diameter             (+ 16 0.3))
(def holes-dist                25)
(def hole-depth                6.5)
(def inner-hole-diameter       (+ 9 0.5))
(def anchor-height             60)
(def anchor-border             5)
(def anchor-border-height      1)
(def nema-gear-diameter        36)
(def nema-support-width        10)
(def screw-hole-diameter       (+ 3 0.5))
(def screw-hole-diameter-tight 3)

(def nema-round-extrusion-height  1.9)
(def nema-length-medium           39)
(def nema-side-size               42.2)
(def gearbox-basis-height         5)
(def gearbox-height               25)
(def gearbox-diameter             36)
(def extruder-height              20)
(def motor-fix-height             10)

(defn round [x & {:keys [d h] :or {d 4}}]
  (minkowski x (cylinder :d d  :h h :center false)))

(defgeometry extruder-block []
  (union (->> (cylinder :d 10 :h 10)
              (rotate [(° 90) 0 0])
              (translate [-5
                          (- (/ gearbox-diameter 2))
                          (- 0 nema-length-medium gearbox-basis-height
                             gearbox-height
                             adapter-height
                             (/ extruder-height 2))])
              -%)
         (->> (cube nema-side-size nema-side-size extruder-height)
              (translate [0 0 (- 0 nema-length-medium gearbox-basis-height
                                 gearbox-height
                                 adapter-height
                                 (/ extruder-height 2))]))))

(defgeometry extruder-motor []
  (->> (motor :model (literal "Nema17")
              :size  (literal "NemaMedium"))
       (translate [0 0 (- (+ nema-length-medium nema-round-extrusion-height))])
       (union (->> (cube nema-side-size nema-side-size gearbox-basis-height)
                   (translate [0 0 (- 0 nema-length-medium
                                      (/ gearbox-basis-height 2))]))
              (->> (cylinder :d gearbox-diameter
                             :h (+ gearbox-height adapter-height))
                   (translate [0 0 (- 0 nema-length-medium gearbox-basis-height
                                      gearbox-height
                                      adapter-height)])))))

(defgeometry center-hole []
  (cylinder :d inner-hole-diameter :h (* 2 effector-height)
            :center true))

(defgeometry two-holes [& {:keys [h] :or {h effector-height}}]
  (let [cyl (cylinder :d hole-diameter :h h :center false)
        ch  (center-hole)
        sh  (->> (cylinder :d      screw-hole-diameter-tight
                           :h      100
                           :center false)
                 (translate [(+ (/ (- hole-diameter inner-hole-diameter) 4)
                                (/ inner-hole-diameter 2)
                                (* 1/3 screw-hole-diameter-tight))
                             0
                             -20]))
        shs (for [a (range 3)]
              (->> sh (rotate [0 0 (° (+ 360/6 (* a 360/3)))])))]
    (->> (union (->> [cyl ch shs])
                (->> [cyl ch shs]
                     (translate [0 holes-dist 0])))
         (translate [0
                     (- (+ (/ hole-diameter 2)
                           (- (/ holes-dist 2)
                              (/ hole-diameter 2))))
                     (- effector-height hole-depth)])
         (rotate [0 0 (° 90)])
         (translate [0 -6 0]))))

(defgeometry effector []
  (let [fill (cylinder :d (+ hole-diameter 5)
                       :h (+ 0.01 effector-height)
                       :center false)]
    (-> (->> (kossel/effector)
             (translate [0 0 (/ effector-height 2)]))
        (union fill)
        (difference (two-holes)))))

(defgeometry screw-holes []
  (let [h  (+ effector-height anchor-height)
        sh (->> (cylinder :d      screw-hole-diameter
                          :h      h
                          :center false)
                (translate [0 12.5 (- (/ h 2))]))]
    (for [a [0 60 180 300]]
      (->> sh (rotate [0 0 (° a)])))))

(defn arc-chord-length [r a]
  (* 2 r (Math/sin (/ a 2))))

(defgeometry extruder-anchor []
  (let [h        anchor-height
        base-h   anchor-border-height
        base-tri (->> (cylinder :d 30 :h base-h :center false :$fn 3)
                      (translate [0 0 (- base-h)]))
        tri-r    (/ (- 25 (* 2 anchor-border)) 2)
        tri-side (arc-chord-length tri-r (° (/ 360 3)))
        tri-h    ; tri-side^2 = (tri-side/2)^2 + h^2
        ; h = √[tri-side^2 - (tri-side/2)^2]
        (√ (- (** tri-side) (** (/ tri-side 2))))
        tri      (->> (cylinder :r tri-r
                                :h h
                                :$fn 3)
                      (translate [0 0 (- h)]))
        place-left       #(->> %1
                               (rotate [0 0 (° (- (+ 90 180)))])
                               (scale [1 1/2 1])
                               (translate [0 %2 0]))
        place-right      #(->> %
                               (rotate [0 0 (° -90)]))
        nema-h           (+ nema-side-size anchor-border-height
                            (- anchor-height (/ nema-side-size 2)))
        nema-placeholder (cube nema-support-width nema-side-size nema-h)
        place-motor      #(->> %
                               (rotate [(° -90) 0 0])
                               (translate [0 nema-length-medium
                                           (- (/ nema-side-size 2))])
                               (rotate [0 0 (° (- 45))])
                               (translate [22 22 (- (* 0.9 anchor-height))]))
        mot              (extruder-motor)
        mot-fix-cube     (->> (cube nema-side-size
                                    nema-side-size
                                    (+ motor-fix-height adapter-height))
                              (translate
                                [0 0
                                 (- 0
                                    nema-length-medium
                                    gearbox-basis-height
                                    gearbox-height
                                    (- (/ (- motor-fix-height adapter-height)
                                          2))
                                    -1)])
                              (>>- (-> (round :d 10 :h 0.1))))
        mot-fix          (->> (difference mot-fix-cube mot)
                              (translate [0 0 (- 0 1 adapter-height)]))
        mot-fix-cube     (->> mot-fix-cube
                              (translate [0 0 (- 0 (/ adapter-height 2))]))
        screw-fix        (->> (extruder-adapter)
                              (rotate [(° 180) 0 0])
                              (translate [0 0
                                          (- 0 nema-length-medium
                                             gearbox-basis-height
                                             gearbox-height
                                             adapter-height)]))]
    (-> (union (-> (union (place-left  base-tri tri-h)
                          (place-right base-tri))
                   (round :h anchor-border-height))
               (->> (union (place-left  tri (/ tri-h 2))
                           (place-right tri))
                    (>>- (-> (round :h anchor-border-height)))
                    (translate [0 -1 0])))
        (difference (two-holes :h (+ 10 effector-height)))
        (difference (->> (screw-holes)
                         (translate [0 0 (- (+ 2 anchor-border-height))])))
        (difference (place-motor (union (extruder-block) mot-fix-cube)))
        (union (->> (union screw-fix mot-fix)
                    (translate [0 0 adapter-height])
                    place-motor))

        (union (place-motor (union mot (extruder-block)))))))

(render ($fn 100 [(effector)
                  #_(extruder-anchor)]))
