(ns open-scad.models.extruder-platform
  (:require [open-scad.core :refer :all]
            [threading.core :refer :all]))

(def hole-diameter          2.8)
(def holes-distance         (- 24.8 5))
(def roundness              5)

(def base-width             30)
(def base-length            15)
(def base-height            1)

(def hook-diameter          10)
(def hook-thickness         1.5)
(def corridor-width         4)

(def platform-length        80)
(def platform-height        2.5)
(def platform-hole-diameter 12)

(def attach-width           15)
(def attach-length          65)
(def attach-thickness       5)
(def attach-holes-distance  (- 55.5 5))
(def hole-attach-distance   30)

(def tip-hole-diameter      7)



(defgeometry ring [r thick]
  (->> (extrude-rotate (translate [r 0 0] (circle thick)))))

(defgeometry equilateral-triangle [length thick]
  ($fn 3 (cylinder (/ length 2) thick :center true)))

(defgeometry hook []
  (let [hole   (union
                 (cylinder (/ hole-diameter 2) (+ 2 base-height) :center true)
                 (->> (cube corridor-width (/ base-length 2) (+ 2 base-height)
                            :center true)
                      (translate [0 (+ (* 1/4 base-length)
                                       (* 1/5 hole-diameter)) 0])))
        hook   (difference
                 (ring (/ hook-diameter 2) hook-thickness)
                 (->> (cube (* 2 hook-diameter)
                            (* 2 hook-diameter)
                            hook-diameter)
                      (translate [(- hook-diameter) 0 0])))]
    (-> (cube base-width base-length base-height :center true)
        (difference
          (apply union (juxt->> hole
                                (translate [(+ (/ holes-distance 2)) 0 0])
                                (translate [(- (/ holes-distance 2)) 0 0]))))
        (union (->> hook
                    (rotate [0 (° -90) 0])
                    (translate [0 0 (/ base-height 2)]))))))

(defgeometry attach []
  (let [screw-hole  (cylinder (/ hole-diameter 2) (* 3 platform-height))]
    (difference
      (cube attach-width attach-length attach-thickness
            :center true)
      (translate [0 (+ (/ attach-holes-distance 2)) 0]
                 screw-hole)
      (translate [0 (- (/ attach-holes-distance 2)) 0]
                 screw-hole))))

(defgeometry platform []
  (let [h           (√ (- (** platform-length) (** (/ platform-length 2))))
        side-circle (cylinder (/ platform-length 2) (* 3 platform-height))
        center-hole (cylinder (/ platform-hole-diameter 2) (* 3 platform-height)
                              :center true)
        tip-hole    (cylinder (/ tip-hole-diameter 2) (* 3 platform-height)
                              :center true)
        d           0.85
        dd          0.57]
    (-> (minkowski (equilateral-triangle platform-length platform-height)
                   (->> ($fn 40 (cylinder roundness platform-height :center true))
                        (translate [0 0 0])))
        (difference (->> (translate [0 (* d h) 0] side-circle)
                         (rotate [0 0 (° (- 90 120))]))
                    (->> (translate [0 (* d h) 0] side-circle)
                         (rotate [0 0 (° (- 90 (* 2 120)))]))
                    (->> (translate [0 (* d h) 0] side-circle)
                         (rotate [0 0 (° (- 90 (* 3 120)))])))
        (difference center-hole)
        (union (->> (attach)
                    (rotate [(° 90) 0 (° (- 90 (* 3 120)))])
                    (translate [(+ (/ attach-thickness 2) hole-attach-distance)
                                0
                                (- (/ (+ attach-length platform-height) 2))])))
        (difference (->> (translate [(* dd h) 0 0] tip-hole)
                         (rotate [0 0 (° 0)]))
                    (->> (translate [(* dd h) 0 0] tip-hole)
                         (rotate [0 0 (° 120)]))
                    (->> (translate [(* dd h) 0 0] tip-hole)
                         (rotate [0 0 (° (* 2 120))]))))))

(render ($fn 40 (union
                  (part :attach (attach))
                  (part :platform (platform))
                  (->> (part :hook (hook))
                       (translate [(* 1.2 platform-length) 0 0])))))
