(ns open-scad.models.valve-actuator
  (:require [open-scad.core :refer :all]
            [open-scad.libs.servo-arm :refer [servo-standard]]
            [open-scad.libs.dot-scad :refer [rounded-cube]]
            [threading.core :refer :all]))

(def mold-offset 0.6)
(def wall-width  2)
(def roundness   1)

;; For a SG90 Micro Servo 9G
(def servo-width                 22.50)
(def servo-length                12.45)
(def servo-height                22.70)

(def servo-base-height           4)
(def servo-base-diameter         11.60)
(def servo-base-kiki-diameter    5.40)

(def servo-cross-width           16.5)
(def servo-cross-side-branch-w   3.75)
(def servo-cross-up-length       18)
(def servo-cross-down-length     16)
(def servo-cross-length          (+ servo-cross-up-length
                                    servo-cross-down-length))
(def servo-cross-height          1.4)


(def valve-diameter              24)
(def valve-height                32)
(def valve-width                 44)
(def side-cylinder-z-offset      5)
(def side-cylinder-diameter      20)
(def side-cylinder-height        2)
(def side-cylinder-depth         1.5)
(def handle-width                4)
(def handle-length               47)
(def handle-height               24)
(def handle-thickness            3)

(def adapter-depth               4.5)
(def adapter-width               (+ (max servo-cross-width  handle-width)
                                    (* 3 wall-width)))
(def adapter-length              (+ (max servo-cross-length handle-length)
                                    (* 2 wall-width)))
(def adapter-height              (+ (max servo-cross-height handle-height)
                                    wall-width
                                    adapter-depth
                                    mold-offset))

(def base-width                  (+ valve-width 10))
(def base-length                 (+ valve-diameter
                                          (* 2 (+ wall-width mold-offset))))
(def base-height                 side-cylinder-z-offset)
(def pilar-width                 5)
(def pilar-length                base-length)
(def pilar-height                (+ base-height
                                    valve-height
                                    adapter-height
                                    #_(* 4 mold-offset)))
(def pilar-base-width            (+ base-width (* 2 pilar-length)))
(def pilar-base-length           base-length)
(def pilar-base-height           base-height)

(def side-slit-width             (+ 2 wall-width))
(def side-slit-length            2)
(def side-slit-height            3)
(def side-base-width             (+ side-cylinder-height
                                          (* 2 wall-width)))
(def side-base-length            base-length)
(def side-base-height            (+ side-cylinder-z-offset
                                    side-cylinder-depth
                                    side-slit-height))

(defgeometry pilar []
  (->> (rounded-cube [pilar-width pilar-length pilar-height]
                     roundness)
       (rotate [0 0 (° 90)])
       (translate [(/ (+ base-width pilar-length) 2)
                   (+ (/ (+ pilar-base-length pilar-base-width)
                         2)
                      (- pilar-width))
                   0])))

(defgeometry pilar-base []
  (->> (rounded-cube [pilar-base-width pilar-base-length pilar-base-height]
                     roundness
                     :center true)
       (rotate [0 0 (° 90)])
       (translate [(/ base-width 2)
                   (/ base-length 2)
                   (/ base-height 2)])
       (union (pilar)
              (translate [0 (+ (- pilar-base-width) pilar-width) 0]
                         (pilar)))))

(defgeometry base-mold []
  (let [r           (/ (+ valve-diameter mold-offset) 2)
        h           (+ side-cylinder-z-offset wall-width 2)
        dig-cyl     (cylinder r h :center false)
        base        (rounded-cube [base-width base-length base-height]
                                  roundness
                                  :center false)
        side-r      (/ side-cylinder-diameter 2)
        side-h      side-cylinder-height
        side-z-off  (- base-height wall-width)
        side-x-off  (- (/ valve-width 2) (/ side-h 2))
        base-center #(translate [(/ base-width  2)
                                 (/ base-length 2)
                                 side-z-off]
                                %)
        side-cyl    (->> (cylinder (+ side-r mold-offset) (+ side-h mold-offset)
                                   :center true)
                         (rotate [0 (° 90) 0])
                         (base-center)
                         (translate [0 0 (+ side-r
                                            side-cylinder-z-offset)]))
        slit-x-off  (- (/ (+ (/ side-slit-width 2) valve-width) 2)
                       (- side-base-width wall-width))
        side-slit   (->> (cube side-slit-width
                               (+ side-slit-length mold-offset)
                               (+ side-slit-height mold-offset))
                         (base-center)
                         (translate [0 0
                                     (+ side-base-height
                                        (/ side-slit-height 2)
                                        (- side-slit-height))]))
        side-base   (->> (cube side-base-width
                               side-base-length
                               side-base-height
                              :center true)
                        (base-center)
                        (translate [0 0 (/ side-base-height 2)])
                        (>>- (-> (difference side-cyl)
                                 #_(union side-slit))))]
    (-> base
        (union (pilar-base))
        (difference (->> (base-center dig-cyl)
                         (translate [0 0 (- (- side-z-off wall-width))])))
        (union (difference
                 (translate [(+ side-x-off) 0 0] side-base)
                 (translate [(+ slit-x-off) 0 0] side-slit))
               (difference
                 (translate [(- side-x-off) 0 0] side-base)
                 (translate [(- slit-x-off) 0 0] side-slit))))))

(defgeometry servo-cross-mold []
  (let [side-off (/ (- servo-cross-width servo-cross-side-branch-w)
                    2)]
    (-> (servo-standard (- servo-cross-up-length 3) 1)
        (union
          (->> (servo-standard (- servo-cross-down-length 3) 1)
               (rotate [0 0 (° 180)]))

          (->> (apply union
                      (cube (- servo-cross-width
                               servo-cross-side-branch-w)
                            servo-cross-side-branch-w
                            2)
                      (->> (cylinder (/ servo-cross-side-branch-w 2)
                                     2)
                           (translate [side-off 0 0])
                           (>>- (juxt-> identity
                                        (->> (rotate [0 0 (° 180)]))))))
               (translate [0 0 5])))
        (union (->> (cylinder 1.1 2 :center false)
                    (translate [0 (- servo-cross-up-length 3) 4]))
               (->> (cylinder 1.1 2 :center false)
                    (translate [0 (- (- servo-cross-down-length 3)) 4]))))))

(defgeometry servo-handle-adapter []
  (difference
    (->> (rounded-cube [adapter-width adapter-length adapter-height]
                       roundness :center true)
         (translate [0 0 (/ adapter-height 2)]))
    (->> (union (->> (servo-cross-mold)
                     (scale [(/ (+ servo-cross-width  (* 3 mold-offset))
                                servo-cross-width)
                             (/ (+ servo-cross-length (* 4 mold-offset))
                                servo-cross-length)
                             10])
                     (rotate [0 (° 180) 0])
                     (translate
                       [0 0 (+ (* 10 (+ 4 2))
                               adapter-height
                               (- (+ adapter-depth mold-offset)))]))
                (->> (cube (+ handle-width  mold-offset)
                           (+ handle-length (* 2 mold-offset))
                           (+ handle-height mold-offset))
                     (translate [0 0 (/ handle-height 2)])))
         (rotate [(° 180) 0 0])
         (translate [0 0 adapter-height]))))

(render ($fn 40
             (part :base                      (base-mold))
             (->> (part :servo-handle-adapter (servo-handle-adapter))
                  (translate [(/ base-width  2)
                              (/ base-length 2)
                              (+ wall-width valve-height handle-thickness)]))))
