(ns open-scad.models.glass-tip-box
  (:require [clojure.java.io :as io]
            [open-scad.core :refer :all :exclude [import]]
            [open-scad.libs.threads :refer [metric-thread]]
            [threading.core :refer :all]))

(def layer-height        0.2)
(def tip-diameter        8)
(def tip-length          40)
(def tip-inset-factor    0.75)
(def n-central-tips      6)
(def box-height          (* 1.5 tip-diameter))
(def box-rim-thickness   2)
(def box-rim-height      (+ 5 layer-height))
(def separator-thickness (- box-rim-height
                            (* 2 (- 1 tip-inset-factor)
                               tip-diameter)))

(def tip-dist          (/ (- (+ tip-length (* 2.5 tip-diameter))
                             (* n-central-tips tip-diameter))
                          n-central-tips))
(def side-off          (* 2 (+ (/ (+ tip-length tip-diameter) 2)
                               tip-dist)))
(def longest-side      (max (+ tip-length (* 2 (+ tip-diameter tip-dist)))
                            (+ (* n-central-tips tip-diameter)
                               (* (dec n-central-tips) tip-dist))))
(def box-diameter      (√ (* 2 (** longest-side)))) ;; d = √(side^2 + side^2)

(defgeometry glass-tip []
  (cylinder (/ tip-diameter 2) tip-length :center true))

(defn centered-range [n]
  (let [m (/ (dec n) 2)]
    (map #(- % m) (range n))))

(defgeometry import [file & {:keys [dpi center]}])

(defgeometry ortie-face []
  (import (.getFile (io/resource "ortie.svg"))
          :center true))

(defgeometry ortie-surf []
  (->> (import (.getFile (io/resource "ortie-surf.svg"))
               :center true)
       (scale [0.09 0.09 0])
       (translate [-2 3 0])))

(defgeometry glass-tip-box [& {:keys [top]}]
  (let [tip       (->> (glass-tip)
                       (rotate [(° 90) 0 0]))

        tips      (for [x (centered-range n-central-tips)]
                    (translate [(* x (+ tip-diameter tip-dist)) 0 0]
                               tip))
        side-tips (for [x [-1 1]]
                    (->> tip
                         (rotate [0 0 (° 90)])
                         (translate [0 (* x (/ side-off 2)) 0])))
        all-tips [tips side-tips]
        move-up (partial translate [0 0 (- (+ (/ tip-diameter 2) box-height)
                                           (* tip-inset-factor tip-diameter))])
        thread  (-> (metric-thread (- box-diameter (* 2 box-rim-thickness))
                                   3 (+ box-rim-height (if top 2 0))
                                   :groove false
                                   :leadin 1
                                   :angle 45
                                   :internal top)
                    (when->> (<<- top) (translate [0 0 -1])))
        cyl     (-> (cylinder (- (/ box-diameter 2)
                                 (if top 0 (* 2 box-rim-thickness)))
                              (+ box-rim-height (if top 0 2))
                              :center false)
                    (when->> (<<- (not top)) (translate [0 0 -1])))
        thread (if top
                 (difference cyl thread)
                 (difference thread cyl))]
    (-> (cylinder (/ box-diameter 2) box-height :center false)
        (difference (move-up all-tips))
        (union (->> thread
                    (translate [0 0 box-height])))
        (difference (->> (if top (ortie-surf) (ortie-face))
                         (extrude-linear {:height (* 2 layer-height)}))))))

(defgeometry indica []
  (->> (import (.getFile (io/resource "indica.svg"))
               :center true)
       (scale [0.15 0.15 0])))

(defgeometry sativa []
  (->> (import (.getFile (io/resource "sativa.svg"))
               :center true)
       (scale [0.20 0.20 0])
       (translate [3 -7 0])
       (rotate [0 0 (° -45)])))

(defgeometry separator []
  (difference (cylinder (* 0.95 (- (/ box-diameter 2) box-rim-thickness))
                        separator-thickness
                        :center false)
              (->> (indica)
                   (extrude-linear {:height (* 2 layer-height)}))
              (->> (sativa)
                   (extrude-linear {:height (* 2 layer-height)})
                   (translate [0 0 separator-thickness]))))

(render ($fn 100  [(->> (part :bottom (glass-tip-box))
                        (translate [+50 0 0]))
                   (->> (part :top (glass-tip-box :top true))
                        (translate [-50 0 0]))
                   (->> (part :separator (separator))
                        (translate [0   65 0]))]))
