(ns open-scad.models.8-channel-relay-box
  (:require [threading.core :refer :all]
            [weaving.core :refer :all]
            [open-scad.core :refer :all]
            [open-scad.libs.dot-scad :refer [rounded-cube]]))

(def mold-offset 0.6)

(def relay-width                52)
(def relay-length               134)
(def relay-height               19)
(def relay-thickness            1.3)
(def wall-width                 5)
(def relay-dist-to-wall         1)
(def roundness                  5)

(def screw-hole-diameter        3)
(def screw-hole-dist-to-edge    1)
(def screw-height               8)
(def screw-diameter             3.2)
(def screw-safety-height        1)

(def string-hole-diameter       4)

(def grounded-cable-diameter    7.4)
(def ungrounded-cable-height    3.75)
(def ungrounded-cable-width     6)
(def ac-bay-dist-to-edge        5.5)
(def ac-bay-width               5)
(def ac-bay-hub-length          29.75)
(def ac-bay-interstice          1.40)
(def ac-bay-nb-ports            8)
(def ac-bay-nb-grounded-ports   4)
(def ac-bay-nb-ungrounded-ports 4)
(def ac-bay-hub-nb-ports        2)


(def gpio-bay-width             2.2)
(def gpio-bay-length            26)
(def gpio-side-width            20.5)
(def gpio-bay-hole-width        gpio-side-width)
(def gpio-bay-dist-to-edge      2.5)

(def screw-tower-diameter       (* 2 screw-hole-diameter))
(def screw-tower-height         (- (+ screw-height screw-safety-height)
                                   wall-width relay-thickness))
(def ext-screw-diameter         screw-diameter)
(def screw-socket-diameter      6)
(def screw-socket-height        3)
(def ext-screw-height           16)

(def box-width         (+ relay-width  (* 2 (+ relay-dist-to-wall wall-width))))
(def box-length        (+ relay-length (* 2 (+ relay-dist-to-wall wall-width))))
(def box-height        (+ relay-height screw-tower-height
                          (* 2 (+ relay-dist-to-wall wall-width))))
(def box-digger-width  (- box-width  (* 2 wall-width)))
(def box-digger-length (- box-length (* 2 wall-width)))
(def box-digger-height (- box-height (* 2 wall-width)))


(defgeometry relay-box-digger []
  (apply cube (concat [box-digger-width box-digger-length box-digger-height]
                      [:center true])))

(defgeometry relay-box []
  (->> (difference
         (rounded-cube [box-width box-length box-height] roundness :center true)
         (relay-box-digger))
       (translate [0 0 (- (* 1/2 box-height)
                          (+ screw-height screw-safety-height))])))

(defgeometry screw-towers []
  (apply union (for [opw [+ -]
                     opl [+ -]]
                 (->> (difference
                        (cylinder (/ screw-tower-diameter 2) screw-tower-height)
                        (cylinder (/ screw-diameter 2) screw-height))
                      (translate [(opw (- (* 1/2 relay-width)
                                          screw-hole-dist-to-edge
                                          (* 1/2 screw-hole-diameter)))
                                  (opl (- (* 1/2 relay-length)
                                          screw-hole-dist-to-edge
                                          (* 1/2 screw-hole-diameter)))
                                  (- (*  1/2 screw-tower-height)
                                     (- (+ screw-height screw-safety-height)
                                        wall-width))])))))

(defgeometry external-screw-sockets []
  (let [screw-socket (cylinder (/ screw-socket-diameter 2)
                               screw-socket-height
                               :center true)
        screw-hole   (cylinder (/ ext-screw-diameter 2)
                               (+ 2 ext-screw-height screw-safety-height)
                               :center true)]
    (apply union (for [opw [+ -]
                       opl [+ -]]
                   (->> (union
                          (translate
                            [(- (opw 1)) (- (opl 1)) 0#_(/ screw-socket-height 2)]
                            screw-socket)
                          (translate
                            [(- (opw 1)) (- (opl 1)) (/ (+ screw-socket-height
                                                           ext-screw-height
                                                           screw-safety-height)
                                                        2)]
                            screw-hole))
                        (translate [(opw (* 1/2 (- box-width  wall-width)))
                                    (opl (* 1/2 (- box-length wall-width)))
                                    (- (*  1/2 screw-socket-diameter)
                                       (+ screw-height screw-safety-height)
                                       2)]))))))

(defgeometry input-ac-hole []
  (->> (cylinder (/ grounded-cable-diameter 2) (+ 2 wall-width))
       (rotate [(° 90) 0 0])
       (translate [(* -1/2 (- relay-width gpio-side-width))
                   (+ 2 (* -1/2 box-length))
                   (- (* 1/2 box-height) (+ screw-height
                                            screw-safety-height))])))

(defgeometry grounded-cable-hole []
  (cylinder (+ (/ grounded-cable-diameter 2) mold-offset) (+ 2 wall-width)))

(defgeometry ungrounded-cable-hole []
  (rounded-cube [(+ ungrounded-cable-width mold-offset)
                 (+ ungrounded-cable-height mold-offset)
                 (+ 2 wall-width)]
                (/ ungrounded-cable-height 2)
                :center true))

(defgeometry ac-bay-holes []
  (let [port-length (/ ac-bay-hub-length ac-bay-hub-nb-ports)
        grounded    (grounded-cable-hole)
        ungrounded  (ungrounded-cable-hole)
        shapes      (concat (repeat ac-bay-nb-grounded-ports   grounded)
                            (repeat ac-bay-nb-ungrounded-ports ungrounded))]
    (->> (apply union
                (for [n (range ac-bay-nb-ports)
                      :let [q (quot n 2)
                            m (mod n 2)
                            s (nth shapes n)
                            ty (+ (* q (+ ac-bay-interstice
                                          ac-bay-hub-length))
                                  (* m port-length)
                                  (/ port-length 2))]]
                  (translate [0 ty 0] s)))
         (rotate [0 (° 90) 0])
         (translate [(/ (+ relay-width (+ 2 wall-width)) 2)
                     (- (- (/ relay-length 2) ac-bay-dist-to-edge))
                     (- (/ box-height 2) (+ screw-height screw-safety-height))]))))

(defgeometry gpio-bay-hole []
  (->> (rounded-cube [gpio-bay-hole-width
                      (+ gpio-bay-length gpio-bay-hole-width)
                      (+ 2 wall-width)]
                     roundness
                     :center true)
       (translate [(- (/ (- relay-width
                            gpio-bay-hole-width
                            gpio-bay-dist-to-edge)
                         2))
                   0
                   (+ (/ box-height 2) wall-width 1)])))

(defgeometry hanging-hole []
  (->> (circle (/ string-hole-diameter 2))
       (translate [roundness 0 0])
       (extrude-rotate {:angle 90})))

(defgeometry hanging-holes []
  (apply union
         (for [ty (map #(% (* 1/2 2/3 box-length)) [+ -])]
           (->> (hanging-hole)
                (rotate [(° -90) 0 0])
                (translate [(* -1/2 box-width)
                            ty
                            (+ (* 1/2 (+ box-height roundness))
                               wall-width
                               1)])))))

(defgeometry relay-bottom []
  (-> (relay-box)
      (difference (->> (cube (* 2 box-width) (* 2 box-length) box-height)
                       (translate [0 0 (/ box-height 2)])))
      (union (screw-towers))
      (difference
        (external-screw-sockets)
        (let [holes
              (->> (hanging-holes)
                   (translate [0 0 (- (+ (* 1/2 (+ box-height roundness))
                                         wall-width
                                         1))])
                   (rotate [(° 180) 0 0])
                   (translate [0 0
                               (- (+ screw-height screw-safety-height 1))]))]
          (union holes
                 (->> holes
                      (rotate [0 0 (° 180)])
                      (translate [box-width 0 0])))))))

(defgeometry relay-top []
  (-> (relay-box)
      (difference (->> (cube (* 2 box-width) (* 2 box-length) box-height)
                       (translate [0 0 (* -1/2 box-height)])))
      (difference (external-screw-sockets)
                  (input-ac-hole)
                  (ac-bay-holes)
                  (gpio-bay-hole)
                  (hanging-holes)
                  (->> (hanging-holes)
                       (rotate [0 0 (° 180)])))))

(defgeometry flex-cap []
  (let [[w l h] [gpio-bay-hole-width
                 (+ gpio-bay-length gpio-bay-hole-width)
                 wall-width]
        hole (difference
               (rounded-cube [w l h] roundness :center true)
               (translate [0 0 (+ wall-width)] (cube w l h :center true))
               (translate [0 0 (- wall-width)] (cube w l h :center true)))
        cap (scale [1.2 1.2 0.2] hole)
        center-hole (->> (cylinder 1 (+ 2 wall-width) :center true)
                         (translate [(* -1/4 w) 0 0]))
        slit (rounded-cube [(* 1/3 w) 0.3 (+ 2 wall-width)] 0.3 :center true)]
    (-> (union (translate [(* 0.05 w) 0 (/ wall-width 2)]  cap)
               hole
               (translate [(* 0.05 w) 0 (/ wall-width -2)] cap))
        (difference center-hole
                    (translate [(- (* (+ 1/8 1/3) w)) 0 0] slit)))))

(render ($fn 100
             (union (->> (part :top (relay-top)))
                    (->> (part :bottom (relay-bottom))
                         (translate [(* 1.5 box-width) 0 0]))
                    (->> (part :flex-cap (flex-cap))))))

