(ns toxi.geom.triangle2d
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math]
    [toxi.geom.core :as geom]
    [toxi.geom.common :as common])
  (:import
    [toxi.geom.types Triangle2D]))

(defn equilateral-triangle2d
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.createEquilateralFrom"}
  ([line] (equilateral-triangle2d (:a line) (:b line)))
  ([a b]
    (let[m (geom/interpolate a b 0.5)
         dir (geom/sub a b)
         n (geom/perpendicular dir)
         c (geom/add m (geom/normalize n (* (geom/mag dir) (* math/SQRT3 0.5))))]
      (geom/triangle2d a b c)))
  ([x1 y1 x2 y2] (equilateral-triangle2d (geom/vec2d x1 y1) (geom/vec2d x2 y2))))

(defn- tri2d-from-barycentric
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.fromBarycentric"}
  [tri bary]
  (let [{:keys [x y z]} bary
        a (:a tri) b (:b tri) c (:c tri)
        px (+ (* (:x a) x) (* (:x b) y) (* (:x c) z))
        py (+ (* (:y a) x) (* (:y b) y) (* (:y c) z))]
    (geom/vec2d px py)))

(defn- tri2d-clockwise?
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.isClockwise"}
  ([tri] (tri2d-clockwise? (:a tri) (:b tri) (:c tri)))
  ([a b c]
    (let [ax (:x a) ay (:y a)
          bx (:x b) by (:y b)
          cx (:x c) cy (:y c)]
      (pos? (- (* (- bx ax) (- cy ay)) (* (- cx ax) (- by ay)))))))

(defn- tri2d-flip
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.flipVertexOrder"}
  [tri]
  (assoc tri :b (:c tri) :c (:b tri)))

(defn- tri2d-centroid
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.computeCentroid"}
  [tri]
  (geom/scale-n (geom/add (:a tri) (:b tri) (:c tri)) 1/3))

(defn- tri2d-contains-point?
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.containsPoint"}
  [tri p]
  (if (or (= p (:a tri)) (= p (:b tri)) (= p (:c tri)))
    true
    (let [n1 (geom/normalize (geom/sub p (:a tri)))
          n2 (geom/normalize (geom/sub p (:b tri)))
          n3 (geom/normalize (geom/sub p (:c tri)))
          total (+ (Math/acos (geom/dot n1 n2))
                  (Math/acos (geom/dot n2 n3))
                  (Math/acos (geom/dot n3 n1)))
          delta (math/abs (- total math/TWO_PI))]
      (<= delta 0.001))))

(defn- tri2d-area
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.getArea"}
  [tri]
  (* 0.5 (geom/cross (geom/sub (:b tri) (:a tri)) (geom/sub (:c tri) (:a tri)))))

(defn- tri2d-circumference
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.getCircumference"}
  [{:keys [a b c]}]
  (+ (+ (geom/distance a b) (geom/distance b c)) (geom/distance c a)))

(defn- tri2d-adjust-size
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.adjustTriangleSizeBy"}
  ([tri offset] (tri2d-adjust-size tri offset offset offset))
  ([{:keys [a b c] :as tri} offab offbc offca]
    (let [centroid (tri2d-centroid tri)
          ab (geom/offset-and-grow-by (geom/line2d a b) offab 100000000 centroid)
          bc (geom/offset-and-grow-by (geom/line2d b c) offab 100000000 centroid)
          ca (geom/offset-and-grow-by (geom/line2d c a) offab 100000000 centroid)]
      (geom/triangle2d
        (:pos (geom/intersect ab ca))
        (:pos (geom/intersect ab bc))
        (:pos (geom/intersect bc ca))))))

(defn- tri2d-circum-circle
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.getCircumCircle"}
  [tri]
  (let [a (:a tri) b (:b tri) c (:c tri)
        cr (geom/cross (geom/bisect a b) (geom/bisect b c))]
    (if (>= (math/abs (:z cr)) math/*epsilon*)
        (let [orig (geom/vec2d (/ (:x cr) (:z cr)) (/ (:y cr) (:z cr)))
              sa (geom/distance a b)
              sb (geom/distance b c)
              sc (geom/distance c a)
              denom (* (+ sa sb sc) (- (+ sb sc) sa) (+ (- sa sb) sc) (- (+ sa sb) sc))
              radius (/ (* sa sb sc) (Math/sqrt denom))]
          (geom/circle orig radius)))))

(defn- tri2d-closest-point-to
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.getClosestPointTo"}
  [tri p]
  (let [a (:a tri) b (:b tri) c (:c tri)
        rab (geom/closest-point (geom/line2d a b) p)
        rbc (geom/closest-point (geom/line2d b c) p)
        rca (geom/closest-point (geom/line2d c a) p)
        dab (geom/mag-squared (geom/sub p rab))
        dbc (geom/mag-squared (geom/sub p rbc))
        dca (geom/mag-squared (geom/sub p rca))
        mind (min dab dbc dca)]
    (cond
      (= mind dab) rab
      (= mind dbc) rbc
      :else rca)))

(defn- tri2d->polygon2d
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.toPolygon2D"}
  [tri]
  (geom/polygon2d [(:a tri) (:b tri) (:c tri)]))

(defn- tri2d-bounds
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.getBounds"}
  [tri]
  (let [a (:a tri) b (:b tri) c (:c tri)
        minx (min (:x a) (:x b) (:x c))
        maxx (max (:x a) (:x b) (:x c))
        miny (min (:y a) (:y b) (:y c))
        maxy (max (:y a) (:y b) (:y c))]
    (geom/rect minx miny (- maxx minx) (- maxy miny))))

(defn- tri2d-random-point
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.getRandomPoint"}
  [tri]
  (let [b1 (math/random)
        b2 (math/random (- 1.0 b1))
        b3 (- 1.0 (+ b1 b2))]
    (tri2d-from-barycentric tri (geom/vec3d (shuffle [b1 b2 b3])))))

(defn- tri2d-edges
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.getEdges"}
  [tri]
  (let [a (:a tri) b (:b tri) c (:c tri)]
    [(geom/line2d a b) (geom/line2d b c) (geom/line2d c a)]))

(defn- tri2d-intersect-triangles
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D.intersectsTriangle"}
  [tri tri2]
  (let [a (:a tri) b (:b tri) c (:c tri)
        a2 (:a tri2) b2 (:b tri2) c2 (:c tri2)]
    (if (or
          (tri2d-contains-point? tri a2)
          (tri2d-contains-point? tri b2)
          (tri2d-contains-point? tri c2)
          (tri2d-contains-point? tri2 a)
          (tri2d-contains-point? tri2 b)
          (tri2d-contains-point? tri2 c))
      true
      (let [ea (geom/edges tri) eb (geom/edges tri2)
            res (loop [la ea]
                  (if-not (nil? (seq la))
                    (if-not
                      (nil?
                        (loop [lb eb]
                          (if-not (nil? (seq lb))
                            (let [type (:type (geom/intersect (first la) (first lb)))]
                              (if (or (= type :intersect) (= type :coincident))
                                true
                                (recur (rest lb)))))))
                      true (recur (rest la)))))]
        (true? res)))))

(defn extend-triangle2d
  [type]
  (extend type
    geom/IIntersectable {
		  :intersect tri2d-intersect-triangles
    }
    geom/IProximity {
		  :closest-point tri2d-closest-point-to
    }
    geom/IShape {
      :bounds tri2d-bounds
      :centroid tri2d-centroid
      :contains-point? tri2d-contains-point?
      :random-point tri2d-random-point
    }
    geom/IShape2D {
      :area tri2d-area
      :bounding-circle tri2d-circum-circle
      :circumference tri2d-circumference
      :edges tri2d-edges
      :->polygon2d tri2d->polygon2d
    }
    geom/ITriangle {
      :adjust-size tri2d-adjust-size
		  :clockwise? tri2d-clockwise?
		  :from-barycentric tri2d-from-barycentric
    }
  ))

(extend-triangle2d Triangle2D)
