(ns toxi.geom.rect
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math]
    [toxi.geom.core :as geom]
    [toxi.math.matrix4x4])
  (:import
    [toxi.geom.types Rect]))

(defn- rect-centroid
  {:added "0.1"
   :java-id "toxi.geom.Rect.getCentroid"}
  [rect]
  (geom/add
    (geom/scale-n (geom/vec2d (:width rect) (:height rect)) 0.5)
    rect))

(defn- rect-area
  {:added "0.1"
   :java-id "toxi.geom.Rect.getArea"}
  [rect] (* (:width rect) (:height rect)))

(defn- rect-circumference
  {:added "0.1"
   :java-id "toxi.geom.Rect.getCircumference"}
  [rect] (* 2 (+ (:width rect) (:height rect))))

(defn- rect-contains-point?
  {:added "0.1"
   :java-id "toxi.geom.Rect.containsPoint"}
  [{:keys [x y width height]} {px :x py :y}]
  (and (>= px x) (< px (+ x width)) (>= py y) (< py (+ y height))))

(defn- rect-top-left
  [rect] (utils/swizzle rect :xy))

(defn- rect-top-right
  [rect] (geom/vec2d (+ (:x rect) (:width rect)) (:y rect)))

(defn- rect-bottom-right
  [rect] (geom/vec2d (+ (:x rect) (:width rect)) (+ (:y rect) (:height rect))))

(defn- rect-bottom-left
  [rect] (geom/vec2d (:x rect) (+ (:y rect) (:height rect))))

(defn- rect-edges
  {:added "0.1"
   :java-id "toxi.geom.Rect.getEdges"}
  [rect]
  (let [a (rect-top-left rect) b (rect-top-right rect)
        c (rect-bottom-right rect) d (rect-bottom-left rect)]
    [(geom/line2d a b) (geom/line2d b c)
     (geom/line2d c d) (geom/line2d c d)]))

(defn- rect->polygon2d
  {:added "0.1"
   :java-id "toxi.geom.Rect.toPolygon2D"}
  [rect]
  (geom/polygon2d [(rect-top-left rect) (rect-top-right rect)
   (rect-bottom-right rect) (rect-bottom-left rect)]))

(defn- rect-random-point
  {:added "0.1"
   :java-id "toxi.geom.Rect.getRandomPoint"}
  [rect]
  (geom/vec2d (math/random (:x rect) (+ (:x rect) (:width rect)))
         (math/random (:y rect) (+ (:y rect) (:height rect)))))

(defn- rect-bounding-circle
  [rect]
  (let[c (rect-centroid rect)
       radius (geom/distance c (rect-top-left rect))]
    (geom/circle c radius)))

(defn- rect-union
  [{:keys[x y width height]} {rx :x ry :y rw :width rh :height}]
  (let[x1 (min x rx)
       y1 (min y ry)
       x2 (max (+ x width) (+ rx rw))
       y2 (max (+ y height) (+ ry rh))
       w (- x2 x1)
       h (- y2 y1)]
    (geom/rect x1 y1 w h)))

(defn- rect-map-point
  [{:keys[x y width height]} p]
  (geom/vec2d (/ (- (:x p) x) width) (/ (- (:y p) y) height)))

(defn- rect-unmap-point
  [{:keys[x y width height]} p]
  (geom/vec2d (+ (* (:x p) width) x) (+ (* (:y p) height) y)))

(defn extend-rect
  [type]
  (extend type
    geom/IShape {
      :bounds identity
      :centroid rect-centroid
      :contains-point? rect-contains-point?
      :random-point rect-random-point
    }
    geom/IShape2D {
      :area rect-area
      :bounding-circle rect-bounding-circle
      :circumference rect-circumference
      :edges rect-edges
      :->polygon2d rect->polygon2d
    }
    geom/IRect {
      :bottom (fn[r] (+ (:y r) (:height r)))
      :bottom-left rect-bottom-left
      :bottom-right rect-bottom-right
      :dimensions (fn[r] (geom/vec2d (:width r) (:height r)))
      :left (fn[r] (:x r))
      :right (fn[r] (+ (:x r) (:width r)))
      :top (fn[r] (:y r))
      :top-left rect-top-left
      :top-right rect-top-right
      :union rect-union
      :map-point rect-map-point
      :unmap-point rect-unmap-point
    }))

(extend-rect Rect)
