(ns toxi.geom.circle
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math]
    [toxi.geom.core :as geom]
    [toxi.geom.vec2d :as v2d]
    [toxi.geom.ellipse])
  (:import
    [toxi.geom.types Circle]))

(def ^{:dynamic true} *circle-res* 20)

(defn- circle-intersect
  {:added "0.1"
   :java-id "toxi.geom.Circle.intersectsCircle"}
  [c1 c2]
  (let [delta (geom/sub c2 c1)
        d (geom/mag delta)
        r1 (:radius c1)
        r2 (:radius c2)]
    (if (and (<= d (+ r1 r2)) (>= d (math/abs (- r1 r2))))
      (let [a (/ (+ (- (* r1 r1) (* r2 r2)) (* d d)) (* 2 d))
            invd (/ 1.0 d)
            p (geom/add (utils/swizzle c1 :xy) (geom/scale-n delta (* a invd)))
            h (Math/sqrt (- (* r1 r1) (* a a)))
            perp (geom/scale-n (geom/perpendicular delta) (* h invd))]
        [(geom/add p perp) (geom/sub p perp)]))))

(defn- circle-tangent-points
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getTangentPoints"}
  [c p]
  (let [m (geom/interpolate c p 0.5)]
    (circle-intersect c (geom/circle m (geom/distance m p)))))

(defn- circle-area
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getArea"}
  [c] (let[r (:radius c)] (* math/PI r r)))

(defn- circle-bounds
  [c] (let [r (geom/vec2d (:radius c))] (geom/rect (geom/sub c r) (geom/add c r))))

(defn- circle-circumference
  {:added "0.1"
   :java-id "toxi.geom.Circle.getCircumference"}
  [c] (* math/TWO_PI (:radius c)))

(defn- circle-contains-point?
  [c p]
  (<= (geom/distance c p) (:radius c)))

(defn- circle-random-point
  [c]
  (geom/add c (v2d/vec2d-from-theta (math/random math/TWO_PI) (math/random (:radius c)))))

(defn- circle->polygon2d
  ([c] (circle->polygon2d c *circle-res*))
  ([c res] (geom/->polygon2d (geom/ellipse c (:radius c)) res)))

(defn- circle-edges
  ([e] (geom/edges (circle->polygon2d e *circle-res*)))
  ([e res] (geom/edges (circle->polygon2d e res))))

(defn extend-circle
  [type]
  (v2d/extend-vec2d type)
  (extend type
    geom/IIntersectable {
      :intersect circle-intersect
    }
    geom/IShape {
      :bounds circle-bounds
      :contains-point? circle-contains-point?
      :random-point circle-random-point
    }
    geom/IShape2D {
      :area circle-area
      :bounding-circle identity
      :circumference circle-circumference
      :edges circle-edges
      :->polygon2d circle->polygon2d
    }
    geom/ICircle {
      :tangent-points circle-tangent-points
    }))

(extend-circle Circle)
