(ns toxi.geom.scenegraph
  (:require
    [toxi.geom.core :as geom]
    [toxi.math.core :as math]
    [toxi.math.matrix4x4 :as mat]))

(defn scene-node
  ([id] (scene-node id nil))
  ([id instance] (scene-node id instance mat/M4X4-IDENTITY))
  ([id instance matrix] (scene-node id instance matrix [] true))
  ([id instance matrix children enabled]
    {:id id :instance instance :matrix matrix :children children :enabled enabled}))

(defn add-child-nodes
  [node & children]
    (assoc node :children (reduce #(conj %1 %2) (:children node) children)))

(defn replicate-node
  [node num offset]
  (let [instance (:instance node)
        mat (:matrix node)
        replica (reduce
                  #(conj %1
                     (scene-node (str (:id node) "-clone" (first %2)) nil
                       (geom/translate (math/matrix4x4) (geom/scale-n (second %2) (inc (first %2))))
                       [node] (:enabled node)))
                  [] (zipmap (range) (repeat num offset)))]
    (scene-node (str (:id node) "-clones") nil mat/M4X4-IDENTITY replica (:enabled node))))

(defn apply-to-node
  ([node f] (apply-to-node node f mat/M4X4-IDENTITY []))
  ([node f parent-mat acc]
  (let [mat (math/matrix-multiply parent-mat (:matrix node))
        instances (if (:instance node)
               (conj acc (f node mat))
               acc)
        instances (if (pos? (count (:children node)))
               (reduce #(apply-to-node %2 f mat %1)
                       instances (:children node))
               instances)]
    instances)))

(defn transformed-node-instances
  [node] (apply-to-node node (fn[node mat] (geom/transform (:instance node) mat))))

(defn node-transformations
  [node] (apply-to-node node (fn[node mat] {:id (:id node) :matrix mat})))

(defn unique-instance-nodes
  ([node] (unique-instance-nodes node {}))
  ([node acc]
    (let [id (:id node)
          children (:children node)
          instances (if (and (:instance node) (nil? (get acc id)))
               (assoc acc id node)
               acc)
          instances (if (pos? (count children))
               (reduce #(unique-instance-nodes %2 %1)
                       instances children)
               instances)]
    instances)))