(ns toxi.geom.ellipse
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math]
    [toxi.geom.core :as geom]
    [toxi.geom.vec2d :as v2d])
  (:import
    [toxi.geom.types Ellipse]))

(def ^{:dynamic true} *ellipse-res* 20)

(defn- ellipse-foci
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getFoci"}
  [e]
  (let [r (:radius e) x (:x e) y (:y e) focus (geom/mag r)]
    (if (> (:x r) (:y r))
      [(geom/vec2d (- x focus) y) (geom/vec2d (+ x focus) y)]
      [(geom/vec2d x (- y focus)) (geom/vec2d x (+ y focus))])))

(defn- ellipse-contains-point?
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.containsPoint"}
  [e p]
  (let [r (:radius e)
        nx (/ (- (:x p) (:x e)) (:x r))
        ny (/ (- (:y p) (:y e)) (:y r))
        nd (+ (* nx nx) (* ny ny))]
    (<= nd 1.0)))

(defn- ellipse-random-point
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getRandomPoint"}
  [e]
  (-> (v2d/vec2d-from-theta (math/random math/TWO_PI))
    (geom/scale (geom/scale-n (:radius e) (math/random 1.0)))
    (geom/add e)))

(defn- ellipse-area
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getArea"}
  [e] (let[r (:radius e)] (* math/PI (:x r) (:y r))))

(defn- ellipse-circumference
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getCircumference"}
  [e] (* math/PI (Math/sqrt (* 0.5 (geom/mag-squared (:radius e))))))

(defn- ellipse-bounds
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getBounds"}
  [e] (geom/rect (geom/sub e (:radius e)) (geom/add e (:radius e))))

(defn- ellipse-bounding-circle
  [e] (geom/circle (:x e) (:y e) (max (:x (:radius e)) (:y (:radius e)))))

(defn- ellipse->polygon2d
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.toPolygon2D"}
  ([e] (ellipse->polygon2d e *ellipse-res*))
  ([e res]
  (let [step (/ math/TWO_PI res) r (:radius e)]
    (geom/polygon2d (reduce (fn[vertices i]
              (conj vertices
                (geom/add (geom/scale (v2d/vec2d-from-theta (* i step)) r) e)))
      [] (range res))))))

(defn- ellipse-edges
  ([e] (geom/edges (ellipse->polygon2d e *ellipse-res*)))
  ([e res] (geom/edges (ellipse->polygon2d e res))))

(defn extend-ellipse
  [type]
  (v2d/extend-vec2d type)
  (extend type
    geom/IShape {
      :bounds ellipse-bounds
      :contains-point? ellipse-contains-point?
      :random-point ellipse-random-point
    }
    geom/IShape2D {
      :area ellipse-area
      :bounding-circle ellipse-bounding-circle
      :circumference ellipse-circumference
      :edges ellipse-edges
      :->polygon2d ellipse->polygon2d
    }
    geom/IEllipse {
      :foci ellipse-foci
    }))

(extend-ellipse Ellipse)