(ns toxi.geom.vec2d
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math]
    [toxi.geom.core :as geom])
  (:import
    [toxi.geom.types Vec2D]))

(defn- add2d
  "Produces the 2D vector sum of two or more vectors.
   Returns a modifed version of a with only the :x & :y components updated.
   Any other keys present remain intact."
  ([a b]
    (assoc a
           :x (+ ^Double (:x a) ^Double (:x b))
           :y (+ ^Double (:y a) ^Double (:y b))))
  ([a b c]
    (if (and (number? b) (number? c))
      (assoc a
             :x (+ ^Double (:x a) b)
             :y (+ ^Double (:x a) c))
      (assoc a
             :x (+ (+ ^Double (:x a) ^Double (:x b)) ^Double (:x c))
             :y (+ (+ ^Double (:y a) ^Double (:y b)) ^Double (:y c)))))
  ([a b c & more]
    (reduce add2d (add2d a b c) more)))

(defn- sub2d
  "Produces the 2D vector difference of two or more vectors.
   Returns a modifed version of a with only the :x & :y components updated.
   Any other keys present remain intact."
  ([a b]
    (assoc a
           :x (- ^Double (:x a) ^Double (:x b))
           :y (- ^Double (:y a) ^Double (:y b))))
  ([a b c]
    (if (and (number? b) (number? c))
      (assoc a
             :x (- ^Double (:x a) b)
             :y (- ^Double (:x a) c))
      (assoc a
             :x (- (- ^Double (:x a) ^Double (:x b)) ^Double (:x c))
             :y (- (- ^Double (:y a) ^Double (:y b)) ^Double (:y c)))))
  ([a b c & more]
    (reduce sub2d (sub2d a b c) more)))

(defn- scale2d
  [a b]
  (assoc a
         :x (* ^Double (:x a) ^Double (:x b))
         :y (* ^Double (:y a) ^Double (:y b))))

(defn- scale2d-n
  [v ^double n]
  (assoc v :x (* ^Double (:x v) n) :y (* ^Double (:y v) n)))

(defn- rotate2d
  [v ^double theta]
  (let [c (Math/cos theta) s (Math/sin theta)
        x ^Double (:x v) y ^Double (:y v)]
    (assoc v
           :x (- (* x c) (* y s))
           :y (+ (* x s) (* y c)))))

(defn- invert2d
  [v]
  (assoc v :x (* ^Double (:x v) -1.0) :y (* ^Double (:y v) -1.0)))

(defn- perp2d
  [v]
  (assoc v :x (* -1.0 ^Double (:y v)) :y (:x v)))

(defn- mag-squared2d
  ^double
  [v]
  (let [x ^Double (:x v) y ^Double (:y v)] (+ (* x x) (* y y))))

(defn- mag2d
  ^double
  [v]
  (Math/sqrt (mag-squared2d v)))

(defn- distance2d
  ^double
  [a b]
  (mag2d (sub2d a b)))

(defn- distance-squared2d
  ^double
  [a b]
  (mag-squared2d (sub2d a b)))

(defn- normalize2d
  ([v]
    (let [m (mag2d v)]
      (if (pos? m)
        (assoc v
               :x (/ ^Double (:x v) m)
               :y (/ ^Double (:y v) m))
        v)))
  ([v ^double len]
    (let [m (/ (mag2d v) len)]
      (if (pos? (math/abs m))
        (assoc v
               :x (/ ^Double (:x v) m)
               :y (/ ^Double (:y v) m))
        v))))

(defn- limit2d
  [v ^double len]
  (if (> (mag-squared2d v) (* len len))
    (normalize2d v len)
    v))

(defn- dot2d
  ^double
  [a b]
  (+ (* ^Double (:x a) ^Double (:x b)) (* ^Double (:y a) ^Double (:y b))))

(defn- cross2d
  ^double
  [a b]
  (- (* ^Double (:x a) ^Double (:y b)) (* ^Double (:y a) ^Double (:x b))))

(defn- heading2d
  ^double
  [v]
  (Math/atan2 ^Double (:y v) ^Double (:x v)))

(defn- angle-between2d
  ^double
  [a b]
  (Math/acos (dot2d a b)))

(defn- angle-between-norm2d
  ^double
  [a b]
  (Math/acos (dot2d (normalize2d a) (normalize2d b))))

(defn- interpolate2d
  ([a b ^double t]
    (interpolate2d a b t math/lerp))
  ([a b ^double t f]
    (assoc a
           :x (f ^Double (:x a) ^Double (:x b) t)
           :y (f ^Double (:y a) ^Double (:y b) t))))

(defn- min2d
  [a b]
  (geom/vec2d
    (min (:x a) (:x b))
    (min (:y a) (:y b))))

(defn- max2d
  [a b]
  (geom/vec2d
    (max (:x a) (:x b))
    (max (:y a) (:y b))))

(defn- reflect2d
  [v r]
  (let [rv (scale2d-n r (* 2.0 (dot2d v r)))] (sub2d rv v)))

(defn- ->polar2d
  [v]
  (assoc v :x (mag2d v) :y (heading2d v)))

(defn- ->cartesian2d
  [v]
  (assoc v
         :x (* ^Double (:x v) (Math/cos ^Double (:y v)))
         :y (* ^Double (:x v) (Math/sin ^Double (:y v)))))

(defn- ->3dxy2d
  [v]
  (utils/swizzle v :xyz))

(defn- ->3dxz2d
  [v]
  (utils/swizzle v :xzy))

(defn- ->3dyz2d
  [v]
  (utils/swizzle v :zxy))

(defn- zerov2d?
  [v]
  (and (< (math/abs (:x v)) math/EPS)
       (< (math/abs (:y v)) math/EPS)))

(defn- bisect2d
  [a b]
  (let [diff (geom/sub a b)
        dotp (geom/dot diff (geom/add a b))]
    (utils/swizzle diff :xyz (* dotp -0.5))))

(defn- slope2d
  ^double
  [v] (/ ^Double (:y v) ^Double (:x v)))

(defn- v2d-delta-equals
  ([a b] (< (math/abs (distance-squared2d a b)) 1e-6))
  ([a b delta] (< (math/abs (distance-squared2d a b)) (* delta delta))))

(defn- abs2d
  [v]
  (assoc v :x (math/abs (:x v)) :y (math/abs (:y v))))

(defn- v2d->vec
  [v] [(:x v) (:y v)])

(defn vec2d-from-theta
  ([^double theta]
    (geom/vec2d (Math/cos theta) (Math/sin theta)))
  ([^double theta ^double len]
    (geom/vec2d (* (Math/cos theta) len) (* (Math/sin theta) len))))

(defn random-vec2d
  ([]
    (normalize2d
      (geom/vec2d
        (math/normalized-random)
        (math/normalized-random))))
  ([^double len]
    (normalize2d
      (geom/vec2d
        (math/normalized-random)
        (math/normalized-random))
      len)))

(defn extend-vec2d
  [type]
  (extend type
    geom/IInvertible {
      :invert invert2d
    }
    geom/IProximity {
      :distance distance2d
      :distance-squared distance-squared2d
    }
    geom/IRotatable {
      :rotate rotate2d
    }
    geom/IScalable {
      :scale scale2d
      :scale-n scale2d-n
    }
    geom/ITranslatable {
      :translate add2d
    }
    geom/IInterpolatable {
      :interpolate interpolate2d
    }
    geom/IHeading2D {
      :direction normalize2d
      :heading heading2d
      :slope slope2d
    }
    geom/IVec {
      :absv abs2d
      :add add2d
      :angle-between angle-between2d
      :angle-between-norm angle-between-norm2d
      :cross cross2d
      :deltav= v2d-delta-equals
      :dot dot2d
      :limit limit2d
      :mag mag2d
      :mag-squared mag-squared2d
      :maxv max2d
      :minv min2d
      :normalize normalize2d
      :reflect reflect2d
      :sub sub2d
      :->cartesian ->cartesian2d
      :->vec v2d->vec
      :zerov? zerov2d?
    }
    geom/IVec2D {
      :bisect bisect2d
      :perpendicular perp2d
      :->3dxy ->3dxy2d
      :->3dxz ->3dxz2d
      :->3dyz ->3dyz2d
      :->polar ->polar2d
    }))

(extend-vec2d Vec2D)
