(ns toxi.geom.polygon2d
  (:require
    [toxi.geom.utils :as utils]
    [toxi.data.collections :as coll :only [successive-pairs]]
    [toxi.math.core :as math]
    [toxi.geom.core :as geom]
    [toxi.geom.common :as common]
    [toxi.geom.linestrip2d]
    [toxi.geom.vec2d]
    [toxi.geom.vec3d]
    [toxi.geom.mesh.core :as mesh]
    [toxi.math.matrix4x4])
  (:import
    [toxi.geom.types Polygon2D]))

(defn- edges
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getEdges"}
  [poly]
  (common/edges* (conj (:vertices poly) (first (:vertices poly)))))

(defn- contains-point?
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.containsPoint"}
  [poly p]
  (if (some #{p} (:vertices poly))
    true
    (let [{:keys [x y]} p
          verts (:vertices poly)
          res (reduce
                (fn[state v]
                  (let [i (:i state) j (nth verts (:j state))
                        xi (:x v) yi (:y v)
                        xj (:x j) yj (:y j)]
                    (if (and
                          (or (and (< yi y) (>= yj y)) (and (< yj y) (>= yi y)))
                          (< (+ xi (* (/ (- y yi) (- yj yi)) (- xj xi))) x))
                      (assoc state :odd (not (:odd state)) :i (inc i) :j i)
                      (assoc state :i (inc i) :j i))))
                {:odd false :i 0 :j (dec (count verts))}
                verts)]
      (:odd res))))

(defn- area
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getArea"}
  [poly]
  (let [verts (:vertices poly)
        a (first verts)
        res (reduce
              (fn [state v]
                {:a (+ (:a state) (geom/cross (:p state) v)) :p v})
              {:a 0 :p a}
              (conj (vec (drop 1 verts)) a))]
    (* 1/2 (:a res))))

(defn- circumference
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getCircumference"}
  [poly]
  (let [verts (:vertices poly)
        a (first verts)
        res (reduce
              (fn [state v]
                {:c (+ (:c state) (geom/distance (:p state) v)) :p v})
              {:c 0 :p a}
              (conj (vec (drop 1 verts)) a))]
    (:c res)))

(defn- centroid
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getCentroid"}
  [poly]
  (let [verts (:vertices poly)
        a (first verts)
        res (reduce
              (fn [state v]
                (let [p (:p state)
                      cross (geom/cross p v)]
                  {:v (geom/add (:v state) (geom/scale-n (geom/add p v) cross))
                   :p v}))
              {:v (geom/vec2d) :p a}
              (conj (vec (drop 1 verts)) a))]
    (geom/scale-n (:v res) (/ 1.0 (* 6 (area poly))))))

(defn- contains-polygon?
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.containsPolygon"}
  [a b]
  (let [verts (:vertices a)
        res
        (loop [v (:vertices b)]
          (if-not (nil? (seq v))
            (if-not (geom/contains-point? a (first v))
              false
              (recur (rest v)))))]
    (nil? res)))

(defn- intersect
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.intersectPolygon"}
  [a b]
  (if (contains-polygon? a b)
    true
    (let [ea (edges a) eb (edges b)
          res (loop [pairs (partition 2
                              (flatten (reduce #(conj %1 (conj (vec (interpose %2 eb)) %2)) () ea)))]
                (if-not (nil? (first pairs))
                  (let [p (first pairs)
                        type (:type (geom/intersect (first p) (second p)))]
                    (if (or (= type :intersect) (= type :coincident))
                      true
                      (recur (rest pairs))))))]
      (true? res))))

(defn- poly-count
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.numVertices"}
  [poly] (count (:vertices poly)))

(defn- poly->linestrip
  [{:keys[vertices]}]
  (geom/linestrip2d (conj vertices (first vertices))))

(defn- point-on-perimeter
  [poly t]
  (geom/point-at (poly->linestrip poly) t))

(defn- poly->segments
  ([poly ^double len] (poly->segments poly len [] true))
  ([poly ^double len coll] (poly->segments poly len coll true))
  ([poly len coll addfirst?]
    (-> (poly->linestrip poly) (geom/->segments len coll addfirst?))))

(defn- sample-uniform
  ([poly ^double len] (-> (poly->linestrip poly) (geom/sample-uniform len [])))
  ([poly ^double len coll] (-> (poly->linestrip poly) (geom/sample-uniform len coll))))

(defn- extrude
  [poly dir amp]
  (-> (poly->linestrip poly) :vertices (common/extrude-vertices dir amp)))

(defn- tesselate
  [poly]
  (let[c (centroid poly)
       vertices (:vertices poly)]
    (reduce
      (fn[mesh [p q]] (mesh/add-faces mesh c q p))
      (mesh/triangle-mesh3d)
      (coll/successive-pairs (conj vertices (first vertices))))))

(defn- closest-point
  [{:keys[vertices]} p]
  (common/closest-point* (conj vertices (first vertices)) p))

(defn- distance-squared
  [poly p]
  (geom/distance-squared (closest-point poly p) p))

(defn- distance
  [poly p]
  (geom/distance (closest-point poly p) p))

(defn extend-polygon2d
  [type]
  (extend type
    geom/IShape {
      :bounds (fn[poly] (common/bounds* (:vertices poly)))
      :contains-point? contains-point?
      :centroid centroid
      :center common/center*
    }
    geom/IRotatable {
      :rotate common/rotate2d*
    }
    geom/IScalable {
      :scale common/scale*
      :scale-n common/scale-n*
    }
    geom/ITranslatable {
      :translate common/translate*
    }
    geom/IIntersectable {
      :intersect intersect
    }
    geom/ITransformable {
      :transform common/transform2d*
    }
    geom/IProximity {
      :closest-point closest-point
      :distance distance
      :distance-squared distance-squared
    }
    geom/IShape2D {
      :area area
      :bounding-circle (fn[poly] (common/bounding-circle* (centroid poly) (:vertices poly)))
      :circumference circumference
      :edges edges
      :point-at point-on-perimeter
      :->polygon2d identity
    }
    geom/IPolygon2D {
      :contains-polygon? contains-polygon?
      :extrude extrude
      :tesselate tesselate
    }
    geom/IPath {
      :conjv (fn[poly coll] (common/conjv* poly coll))
      :->segments poly->segments
      :sample-uniform sample-uniform
      :vertex-count poly-count
    }
    geom/IFlippable {
      :flip common/flip*
    }))

(extend-polygon2d Polygon2D)
