(ns toxi.geom.aabb
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math]
    [toxi.geom.core :as geom]
    [toxi.geom.mesh.core :as mesh]
    [toxi.geom.vec3d :as v3d])
  (:import
    [toxi.geom.types AABB]))

(defn aabb-from-minmax
  [mi mx]
  (geom/aabb (geom/interpolate mi mx 0.5)
        (geom/scale-n
          (geom/vec3d (- (:x mx) (:x mi))
                 (- (:y mx) (:y mi))
                 (- (:z mx) (:z mi)))
          0.5)))

(defn- aabb-min
  {:added "0.1"
   :java-id "toxi.geom.AABB.getMin"}
  [aabb] (utils/swizzle (geom/sub aabb (:extent aabb)) :xyz))

(defn- aabb-max
  {:added "0.1"
   :java-id "toxi.geom.AABB.getMin"}
  [aabb] (utils/swizzle (geom/add aabb (:extent aabb)) :xyz))

(defn- aabb-contains-point?
  {:added "0.1"
   :java-id "toxi.geom.AABB.containsPoint"}
  [aabb p]
  (let [min (aabb-min aabb) max (aabb-max aabb)
        x1 (:x min) y1 (:y min) z1 (:z min)
        x2 (:x max) y2 (:y max) z2 (:z max)
        x (:x p) y (:y p) z (:z p)]
    (and (>= x x1) (>= y y1) (>= z z1) (<= x x2) (<= y y2) (<= z z2))))

(defn- aabb-random-point
  [aabb]
  (let [mi (aabb-min aabb) mx (aabb-max aabb)]
    (geom/vec3d (math/random (:x mi) (:x mx))
           (math/random (:y mi) (:y mx))
           (math/random (:z mi) (:z mx)))))

(defn- aabb-centroid
  [aabb] (utils/swizzle aabb :xyz))

(defn- aabb->mesh
  [aabb]
  (let [a (aabb-min aabb) g (aabb-max aabb)
        b (geom/vec3d (:x a) (:y a) (:z g))
        c (geom/vec3d (:x g) (:y a) (:z g))
        d (geom/vec3d (:x g) (:y a) (:z a))
        e (geom/vec3d (:x a) (:y g) (:z a))
        f (geom/vec3d (:x a) (:y g) (:z g))
        h (geom/vec3d (:x g) (:y g) (:z a))
        ua (geom/vec2d 0 0) ub (geom/vec2d 1 0)
        uc (geom/vec2d 1 1) ud (geom/vec2d 0 1)]
    (mesh/add-faces (mesh/triangle-mesh3d)
       [[a b f {:uv [ud uc ub]}] [a f e {:uv [ud ub ua]}] ;left
        [b c g {:uv [ud uc ub]}] [b g f {:uv [ud ub ua]}] ;front
        [c d h {:uv [ud uc ub]}] [c h g {:uv [ud ub ua]}] ;right
        [d a e {:uv [ud uc ub]}] [d e h {:uv [ud ub ua]}] ;back
        [e f h {:uv [ua ud ub]}] [f g h {:uv [ud uc ub]}] ;top
        [a d b {:uv [ud uc ua]}] [b d c {:uv [ua uc ub]}]]))) ;bottom

(defn- aabb-bsphere
  [aabb]
  (geom/sphere (utils/swizzle aabb :xyz) (geom/mag (:extent aabb))))

(defn extend-aabb
  [type]
  (v3d/extend-vec3d type)
  (extend type
    geom/IShape {
      :bounds identity
      :centroid aabb-centroid
      :contains-point? aabb-contains-point?
      :random-point aabb-random-point
    }
    geom/IShape3D {
      :bounding-sphere aabb-bsphere
      :->mesh aabb->mesh
    }
    geom/IAABB {
      :bounds-min aabb-min
      :bounds-max aabb-max
    }
  ))

(extend-aabb AABB)
