(ns toxi.geom.vec3d
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math])
  (:use
    [toxi.geom.protocols]
    [toxi.geom.core :only [vec2d vec3d]])
  (:import
    [toxi.geom.types Vec3D]))

(defn- add3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.add"}
  [a b] (assoc a
          :x (+ (:x a) (:x b))
          :y (+ (:y a) (:y b))
          :z (+ (:z a) (:z b))))

(defn- sub3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.sub"}
  [a b] (assoc a
          :x (- (:x a) (:x b))
          :y (- (:y a) (:y b))
          :z (- (:z a) (:z b))))

(defn- scale3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.scale"}
  [a b] (assoc a
          :x (* (:x a) (:x b))
          :y (* (:y a) (:y b))
          :z (* (:z a) (:z b))))

(defn- scale3d-n
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.scale"}
  [a ^double n] (assoc a
                  :x (* (:x a) n)
                  :y (* (:y a) n)
                  :z (* (:z a) n)))

(defn- invert3d
  [v] (scale3d-n v -1))

(defn- cross3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.cross"}
  [a b]
  (let [ax (:x a) ay (:y a) az (:z a)
        bx (:x b) by (:y b) bz (:z b)]
    (vec3d
      (- (* ay bz) (* by az))
      (- (* az bx) (* bz ax))
      (- (* ax by) (* bx ay)))))

(defn- mag-squared3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.magSquared"}
  ^double
  [v] (let [x (:x v) y (:y v) z (:z v)]
        (+ (* x x) (* y y) (* z z))))

(defn- mag3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.magnitude"}
  ^double
  [v] (Math/sqrt (mag-squared3d v)))

(defn- distance3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.distanceTo"}
  ^double
  [a b] (mag3d (sub3d a b)))

(defn- distance-squared3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.distanceToSquared"}
  ^double
  [a b] (mag-squared3d (sub3d a b)))

(defn- normalize3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.getNormalized"}
  ([v] (let [m (mag3d v)]
        (if (> m 0.0) (assoc v
                             :x (/ (:x v) m)
                             :y (/ (:y v) m)
                             :z (/ (:z v) m))
          v)))
  ([v ^double len] (let [m (/ (mag3d v) len)]
                    (if (> (math/abs m) 0.0)
                      (assoc v
                             :x (/ (:x v) m)
                             :y (/ (:y v) m)
                             :z (/ (:z v) m))
                      v))))

(defn- limit3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.getLimited"}
  [v ^double len] (if (> (mag-squared3d v) (* len len))
                    (normalize3d v len)
                    v))

(defn- dot3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.dot"}
  ^double
  [a b] (+ (* (:x a) (:x b)) (* (:y a) (:y b)) (* (:z a) (:z b))))

(defn- angle-between3d
  ^double
  [a b] (Math/acos (dot3d a b)))

(defn- angle-between-norm3d
  ^double
  [a b] (Math/acos (dot3d (normalize3d a) (normalize3d b))))

(defn- min3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.min"}
  [a b]
  (vec3d
    (min (:x a) (:x b))
    (min (:y a) (:y b))
    (min (:z a) (:z b))))

(defn- max3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.max"}
  [a b]
  (vec3d
    (max (:x a) (:x b))
    (max (:y a) (:y b))
    (max (:z a) (:z b))))

(defn- interpolate3d
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.interpolateTo"}
  ([a b ^double t]
    (interpolate3d a b t math/lerp))
  ([a b ^double t f]
    (assoc a
           :x (f ^Double (:x a) ^Double (:x b) t)
           :y (f ^Double (:y a) ^Double (:y b) t)
           :z (f ^Double (:z a) ^Double (:z b) t))))

(defn- reflect3d
  [v r]
  (sub3d (scale3d r (* (dot3d v r) 2)) v))

(defn- ->cartesian3d
  [v]
  (let [x (:x v) y (:y v) z (:z v)
        a (* x (Math/cos z))
        xx (* a (Math/cos y))
        yy (* x (Math/sin z))
        zz (* a (Math/sin y))]
    (assoc v :x xx :y yy :z zz)))

(defn- zerov3d?
  {:added "0.1"
   :java-id "toxi.geom.Vec3D.isZeroVector"}
  [v]
  (and (< (math/abs (:x v)) math/EPS)
       (< (math/abs (:y v)) math/EPS)
       (< (math/abs (:z v)) math/EPS)))

(defn- v3d->2dxy
  [v] (utils/swizzle v :xy))

(defn- v3d->2dxz
  [v] (utils/swizzle v :xz))

(defn- v3d->2dyz
  [v] (utils/swizzle v :yz))

(defn- v3d-delta-equals
  ([a b] (v3d-delta-equals a b 0.001))
  ([a b delta] (< (math/abs (distance-squared a b)) (* delta delta))))

(defn- abs3d
  [v]
  (assoc v :x (math/abs (:x v)) :y (math/abs (:y v)) :z (math/abs (:z v))))

(defn- v3d->vec
  [v] [(:x v) (:y v) (:z v)])

(defn- v3d-rotate-axis
  [v axis theta]
  (let [x (:x v) y (:y v) z (:z v)
        ax (:x axis) ay (:y axis) az (:z axis)
        ux (* ax x) uy (* ax y) uz (* ax z)
        vx (* ay x) vy (* ay y) vz (* ay z)
        wx (* az x) wy (* az y) wz (* az z)
        si (Math/sin theta) co (Math/cos theta)
        uvw (+ ux vy wz)]
    (assoc v
           :x (+ (+ (* ax uvw)
                    (* co (- (* x (+ (* ay ay) (* az az)))
                             (* ax (+ vy wz)))))
                 (* si (+ (* -1 wy) vz)))
           :y (+ (+ (* ay uvw)
                    (* co (- (* y (+ (* ax ax) (* az az)))
                             (* ay (+ ux wz)))))
                 (* si (- wx uz)))
           :z (+ (+ (* az uvw)
                    (* co (- (* z (+ (* ax ax) (* ay ay)))
                             (* az (+ ux vy)))))
                 (* si (+ (* -1 vx) uy))))))

(defn- v3d-rotate-x
  [v theta]
  (let[si (Math/sin theta) co (Math/cos theta)
       y (:y v) z (:z v)]
    (assoc v
           :y (+ (* si z) (* co y))
           :z (- (* co z) (* si y)))))

(defn- v3d-rotate-y
  [v theta]
  (let[si (Math/sin theta) co (Math/cos theta)
       x (:x v) z (:z v)]
    (assoc v
           :x (- (* co x) (* si z))
           :z (+ (* co z) (* si x)))))

(defn- v3d-rotate-z
  [v theta]
  (let[theta (* -1 theta)
       si (Math/sin theta) co (Math/cos theta)
       x (:x v) y (:y v)]
    (assoc v
           :x (- (* co x) (* si y))
           :y (+ (* si x) (* co y)))))

(defn extend-vec3d
  [type]
  (extend type
    IInvertible {
	    :invert invert3d
    }
    IRotatable {
      :rotate-around-axis v3d-rotate-axis
      :rotate-x v3d-rotate-x
      :rotate-y v3d-rotate-y
      :rotate-z v3d-rotate-z
    }
    IScalable {
	    :scale scale3d
	    :scale-n scale3d-n
    }
    IInterpolatable {
      :interpolate interpolate3d
    }
    IVec {
      :absv abs3d
	    :add add3d
	    :angle-between angle-between3d
	    :angle-between-norm angle-between-norm3d
	    :cross cross3d
      :deltav= v3d-delta-equals
	    :distance distance3d
	    :distance-squared distance-squared3d
	    :dot dot3d
	    :limit limit3d
	    :mag mag3d
	    :mag-squared mag-squared3d
	    :maxv max3d
	    :minv min3d
	    :normalize normalize3d
	    :reflect reflect3d
	    :sub sub3d
	    :->cartesian ->cartesian3d
      :->vec v3d->vec
	    :zerov? zerov3d?
    }
    IVec3D {
      :->2dxy v3d->2dxy
      :->2dxz v3d->2dxz
      :->2dyz v3d->2dyz
    }))

(extend-vec3d Vec3D)
