(ns toxi.geom.vec2d
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math])
  (:use
    [toxi.geom.protocols]
    [toxi.geom.core :only [vec2d vec3d]])
  (:import
    [toxi.geom.types Vec2D]))

;; hidden Vec2D implementation of IVec & IVec2D protocols
;; these are mapped to the actual interface method names further below

(defn- add2d
  "Produces the 2D vector sum of two or more vectors.
   Returns a modifed version of a with only the :x & :y components updated.
   Any other keys present remain intact."
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.add"}
  ([a b]
    (assoc a
           :x (+ ^Double (:x a) ^Double (:x b))
           :y (+ ^Double (:y a) ^Double (:y b))))
  ([a b c]
    (assoc a
           :x (+ (+ ^Double (:x a) ^Double (:x b)) ^Double (:x c))
           :y (+ (+ ^Double (:y a) ^Double (:y b)) ^Double (:y c))))
  ([a b c & more]
    (reduce add2d (add2d a b c) more)))

(defn- sub2d
  "Produces the 2D vector difference of two or more vectors.
   Returns a modifed version of a with only the :x & :y components updated.
   Any other keys present remain intact."
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.add"}
  ([a b]
    (assoc a
           :x (- ^Double (:x a) ^Double (:x b))
           :y (- ^Double (:y a) ^Double (:y b))))
  ([a b & more]
    (reduce sub2d (sub2d a b) more)))

(defn- scale2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.scale"}
  [a b]
  (assoc a
         :x (* ^Double (:x a) ^Double (:x b))
         :y (* ^Double (:y a) ^Double (:y b))))

(defn- scale2d-n
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.scale"}
  [v ^double n]
  (assoc v :x (* ^Double (:x v) n) :y (* ^Double (:y v) n)))

(defn- rotate2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.getRotated"}
  [v ^double theta]
  (let [c (Math/cos theta) s (Math/sin theta)
        x ^Double (:x v) y ^Double (:y v)]
    (assoc v
           :x (- (* x c) (* y s))
           :y (+ (* x s) (* y c)))))

(defn- invert2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.getInverted"}
  [v]
  (assoc v :x (* ^Double (:x v) -1.0) :y (* ^Double (:y v) -1.0)))

(defn- perp2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.getPerpendicular"}
  [v]
  (assoc v :x (* -1.0 ^Double (:y v)) :y (:x v)))

(defn- mag-squared2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.magSquared"}
  ^double
  [v]
  (let [x ^Double (:x v) y ^Double (:y v)] (+ (* x x) (* y y))))

(defn- mag2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.magnitude"}
  ^double
  [v]
  (Math/sqrt (mag-squared2d v)))

(defn- distance2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.distanceTo"}
  ^double
  [a b]
  (mag2d (sub a b)))

(defn- distance-squared2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.distanceToSquared"}
  ^double
  [a b]
  (mag-squared2d (sub a b)))

(defn- normalize2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.getNormalized"}
  ([v]
    (let [m (mag2d v)]
      (if (pos? m)
        (assoc v
               :x (/ ^Double (:x v) m)
               :y (/ ^Double (:y v) m))
        v)))
  ([v ^double len]
    (let [m (/ (mag2d v) len)]
      (if (> (math/abs m) 0.0)
        (assoc v
               :x (/ ^Double (:x v) m)
               :y (/ ^Double (:y v) m))
        v))))

(defn- limit2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.getLimited"}
  [v ^double len]
  (if (> (mag-squared2d v) (* len len))
    (normalize2d v len)
    v))

(defn- dot2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.dot"}
  ^double
  [a b]
  (+ (* ^Double (:x a) ^Double (:x b)) (* ^Double (:y a) ^Double (:y b))))

(defn- cross2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.cross"}
  ^double
  [a b]
  (- (* ^Double (:x a) ^Double (:y b)) (* ^Double (:y a) ^Double (:x b))))

(defn- heading2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.heading"}
  ^double
  [v]
  (Math/atan2 ^Double (:y v) ^Double (:x v)))

(defn- angle-between2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.angleBetween"}
  ^double
  [a b]
  (Math/acos (dot a b)))

(defn- angle-between-norm2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.angleBetween"}
  ^double
  [a b]
  (Math/acos (dot (normalize2d a) (normalize2d b))))

(defn- interpolate2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.interpolateTo"}
  ([a b ^double t]
    (interpolate2d a b t math/lerp))
  ([a b ^double t f]
    (assoc a
           :x (f ^Double (:x a) ^Double (:x b) t)
           :y (f ^Double (:y a) ^Double (:y b) t))))

(defn- min2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.min"}
  [a b]
  (vec2d
    (min (:x a) (:x b))
    (min (:y a) (:y b))))

(defn- max2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.max"}
  [a b]
  (vec2d
    (max (:x a) (:x b))
    (max (:y a) (:y b))))

(defn- reflect2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.getReflected"}
  [v r]
  (let [rv (scale-n r (* 2.0 (dot v r)))] (sub rv v)))

(defn- ->polar2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.toPolar"}
  [v]
  (assoc v :x (mag2d v) :y (heading2d v)))

(defn- ->cartesian2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.toCartesian"}
  [v]
  (assoc v
         :x (* ^Double (:x v) (Math/cos ^Double (:y v)))
         :y (* ^Double (:x v) (Math/sin ^Double (:y v)))))

(defn- ->3dxy2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.to3DXY"}
  [v]
  (utils/swizzle v :xyz))

(defn- ->3dxz2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.to3DXZ"}
  [v]
  (utils/swizzle v :xzy))

(defn- ->3dyz2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.to3DYZ"}
  [v]
  (utils/swizzle v :zxy))

(defn- zerov2d?
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.isZeroVector"}
  [v]
  (and (< (math/abs (:x v)) math/EPS)
       (< (math/abs (:y v)) math/EPS)))

(defn- bisect2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.bisect"}
  [a b]
  (let [diff (sub a b)
        dotp (dot diff (add a b))]
    (utils/swizzle diff :xyz (* dotp -0.5))))

(defn- slope2d
  ^double
  [v] (/ ^Double (:y v) ^Double (:x v)))

(defn- v2d-delta-equals
  ([a b] (v2d-delta-equals a b 0.001))
  ([a b delta] (< (math/abs (distance-squared a b)) (* delta delta))))

(defn- abs2d
  [v]
  (assoc v :x (math/abs (:x v)) :y (math/abs (:y v))))

(defn- v2d->vec
  [v] [(:x v) (:y v)])

(defn vec2d-from-theta
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.fromTheta"}
  ([^double theta]
    (vec2d (Math/cos theta) (Math/sin theta)))
  ([^double theta ^double len]
    (vec2d (* (Math/cos theta) len) (* (Math/sin theta) len))))

(defn random-vec2d
  {:added "0.1"
   :java-id "toxi.geom.Vec2D.randomVector"}
  ([]
    (normalize2d
      (vec2d
        (math/normalized-random)
        (math/normalized-random))))
  ([^double len]
    (normalize2d
      (vec2d
        (math/normalized-random)
        (math/normalized-random))
      len)))

(defn extend-vec2d
  [type]
  (extend type
    IInvertible {
      :invert invert2d
    }
    IRotatable {
      :rotate rotate2d
    }
    IScalable {
      :scale scale2d
      :scale-n scale2d-n
    }
    IInterpolatable {
      :interpolate interpolate2d
    }
    IVec {
      :absv abs2d
      :add add2d
      :angle-between angle-between2d
      :angle-between-norm angle-between-norm2d
      :cross cross2d
      :deltav= v2d-delta-equals
      :distance distance2d
      :distance-squared distance-squared2d
      :dot dot2d
      :limit limit2d
      :mag mag2d
      :mag-squared mag-squared2d
      :maxv max2d
      :minv min2d
      :normalize normalize2d
      :reflect reflect2d
      :sub sub2d
      :->cartesian ->cartesian2d
      :->vec v2d->vec
      :zerov? zerov2d?
    }
    IVec2D {
      :bisect bisect2d
      :heading heading2d
      :perpendicular perp2d
      :slope slope2d
      :->3dxy ->3dxy2d
      :->3dxz ->3dxz2d
      :->3dyz ->3dyz2d
      :->polar ->polar2d
    }))

(extend-vec2d Vec2D)
