(ns toxi.geom.rect
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math])
  (:use
    [toxi.geom.protocols]
    [toxi.geom.core :only [vec2d line2d rect circle polygon2d]]
    [toxi.geom.vec2d])
  (:import
    [toxi.geom.types Rect]))

(defn- rect-centroid
  {:added "0.1"
   :java-id "toxi.geom.Rect.getCentroid"}
  [rect]
  (add
    (scale-n (vec2d (:width rect) (:height rect)) 0.5)
    rect))

(defn- rect-area
  {:added "0.1"
   :java-id "toxi.geom.Rect.getArea"}
  [rect] (* (:width rect) (:height rect)))

(defn- rect-circumference
  {:added "0.1"
   :java-id "toxi.geom.Rect.getCircumference"}
  [rect] (* 2 (+ (:width rect) (:height rect))))

(defn- rect-contains-point?
  {:added "0.1"
   :java-id "toxi.geom.Rect.containsPoint"}
  [{:keys [x y width height]} p]
  (let [px (:x p) py (:y p)]
    (and (>= px x) (< px (+ x width)) (>= py y) (< py (+ y height)))))

(defn- rect-top-left
  [rect] (utils/swizzle rect :xy))

(defn- rect-top-right
  [rect] (vec2d (+ (:x rect) (:width rect)) (:y rect)))

(defn- rect-bottom-right
  [rect] (vec2d (+ (:x rect) (:width rect)) (+ (:y rect) (:height rect))))

(defn- rect-bottom-left
  [rect] (vec2d (:x rect) (+ (:y rect) (:height rect))))

(defn- rect-edges
  {:added "0.1"
   :java-id "toxi.geom.Rect.getEdges"}
  [rect]
  (let [a (rect-top-left rect) b (rect-top-right rect)
        c (rect-bottom-right rect) d (rect-bottom-left rect)]
    [(line2d a b) (line2d b c)
     (line2d c d) (line2d c d)]))

(defn- rect->polygon2d
  {:added "0.1"
   :java-id "toxi.geom.Rect.toPolygon2D"}
  [rect]
  (polygon2d [(rect-top-left rect) (rect-top-right rect)
   (rect-bottom-right rect) (rect-bottom-left rect)]))

(defn- rect-random-point
  {:added "0.1"
   :java-id "toxi.geom.Rect.getRandomPoint"}
  [rect]
  (vec2d (math/random (:x rect) (+ (:x rect) (:width rect)))
         (math/random (:y rect) (+ (:y rect) (:height rect)))))

(defn rect-bounding-circle
  [rect]
  (let[c (rect-centroid rect)
       radius (distance c (rect-top-left rect))]
    (circle c radius)))

(defn extend-rect
  [type]
  (extend type
    IShape {
      :bounds identity
      :centroid rect-centroid
      :contains-point? rect-contains-point?
      :random-point rect-random-point
    }
    IShape2D {
      :area rect-area
      :bounding-circle rect-bounding-circle
      :circumference rect-circumference
      :edges rect-edges
      :->polygon2d rect->polygon2d
    }
    IRect {
      :bottom (fn[r] (+ (:y r) (:height r)))
      :bottom-left rect-bottom-left
      :bottom-right rect-bottom-right
      :left (fn[r] (:x r))
      :right (fn[r] (+ (:x r) (:width r)))
      :top (fn[r] (:y r))
      :top-left rect-top-left
      :top-right rect-top-right
      :dimensions (fn[r] (vec2d (:width r) (:height r)))
    }))

(extend-rect Rect)
