(ns toxi.geom.polygon2d
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math])
  (:use
    [toxi.geom.protocols]
    [toxi.geom.core :only [vec2d line2d rect circle polygon2d]]
    [toxi.geom.vec2d])
  (:import
    [toxi.geom.types Polygon2D]))

(defn- poly-add-vertex
  ([poly v]
    (if (vector? v)
      (reduce poly-add-vertex poly v)
      (assoc poly :vertices (conj (:vertices poly) v)))))

(defn- poly-edges
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getEdges"}
  [poly]
  (reduce
    (fn[acc v]
      (if (vector? acc)
        (conj acc (line2d (:b (last acc)) v))
        [(line2d acc v)]))
    (conj (:vertices poly) (first (:vertices poly)))))

(defn- poly-contains-point?
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.containsPoint"}
  [poly p]
  (if (some #{p} (:vertices poly))
    true
    (let [{:keys [x y]} p
          verts (:vertices poly)
          res (reduce
                (fn[state v]
                  (let [i (:i state) j (nth verts (:j state))
                        xi (:x v) yi (:y v)
                        xj (:x j) yj (:y j)]
                    (if (and
                          (or (and (< yi y) (>= yj y)) (and (< yj y) (>= yi y)))
                          (< (+ xi (* (/ (- y yi) (- yj yi)) (- xj xi))) x))
                      (assoc state :odd (not (:odd state)) :i (inc i) :j i)
                      (assoc state :i (inc i) :j i))))
                {:odd false :i 0 :j (dec (count verts))}
                verts)]
      (:odd res))))

(defn- poly-bounds
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getBounds"}
  [poly]
  (let [b (reduce
            (fn[state v]
              (let [{:keys [x y]} v]
                (assoc state
                  :min (vec2d (min (:x (:min state)) x) (min (:y (:min state)) y))
                  :max (vec2d (max (:x (:max state)) x) (max (:y (:max state)) y)))))
            {:min (vec2d (Double/MAX_VALUE) (Double/MAX_VALUE))
             :max (vec2d (Double/MIN_VALUE) (Double/MIN_VALUE))}
            (:vertices poly))]
    (rect (:min b) (:max b))))

(defn- poly-area
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getArea"}
  [poly]
  (let [verts (:vertices poly)
        a (first verts)
        res (reduce
              (fn [state v]
                {:a (+ (:a state) (cross (:p state) v)) :p v})
              {:a 0 :p a}
              (conj (vec (drop 1 verts)) a))]
    (* 1/2 (:a res))))

(defn- poly-circumference
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getCircumference"}
  [poly]
  (let [verts (:vertices poly)
        a (first verts)
        res (reduce
              (fn [state v]
                {:c (+ (:c state) (distance (:p state) v)) :p v})
              {:c 0 :p a}
              (conj (vec (drop 1 verts)) a))]
    (:c res)))

(defn- poly-centroid
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.getCentroid"}
  [poly]
  (let [verts (:vertices poly)
        a (first verts)
        res (reduce
              (fn [state v]
                (let [p (:p state)
                      cross (cross p v)]
                  {:v (add (:v state) (scale-n (add p v) cross))
                   :p v}))
              {:v (vec2d) :p a}
              (conj (vec (drop 1 verts)) a))]
    (scale-n (:v res) (/ 1.0 (* 6 (area poly))))))

(defn- poly-transform
  {:added "0.1"}
  [poly matrix]
  (assoc poly
     :vertices (reduce #(conj %1 (transform-point matrix %2)) [] (:vertices poly))))

(defn- poly-apply-transform
  {:added "0.1"}
  [poly f arg]
  (assoc poly
     :vertices (reduce (fn[state v] (conj state (f v arg))) [] (:vertices poly))))

(defn- poly-scale
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.scale"}
  [poly s]
    (poly-apply-transform poly scale s))

(defn- poly-scale-n
  [poly s]
    (poly-apply-transform poly scale-n s))

(defn- poly-translate
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.translate"}
  [poly t]
    (poly-apply-transform poly add t))

(defn- poly-rotate
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.rotate"}
  [poly theta]
  (poly-apply-transform poly rotate theta))

(defn- poly-flip
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.flipVertexOrder"}
  [poly]
  (assoc poly :vertices (reverse (:vertices poly))))

(defn- poly-contains-polygon?
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.containsPolygon"}
  [a b]
  (let [verts (:vertices a)
        res
        (loop [v (:vertices b)]
          (if-not (nil? (seq v))
            (if-not (contains-point? a (first v))
              false
              (recur (rest v)))))]
    (nil? res)))

(defn- poly-intersect-polygon
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.intersectPolygon"}
  [a b]
  (if (poly-contains-polygon? a b)
    true
    (let [ea (edges a) eb (edges b)
          res (loop [pairs (partition 2
                              (flatten (reduce #(conj %1 (conj (vec (interpose %2 eb)) %2)) () ea)))]
                (if-not (nil? (first pairs))
                  (let [p (first pairs)
                        type (:type (intersect-lines (first p) (second p)))]
                    (if (or (= type :intersect) (= type :coincident))
                      true
                      (recur (rest pairs))))))]
      (true? res))))

(defn- poly-count
  {:added "0.1"
   :java-id "toxi.geom.Polygon2D.numVertices"}
  [poly] (count (:vertices poly)))

(defn- poly-bounding-circle
  [poly]
  (let[c (poly-centroid poly)
       radius (reduce #(max %1 (distance c %2)) 0 (:vertices poly))]
    (circle c radius)))

(defn extend-polygon2d
  [type]
  (extend type
    IShape {
      :bounds poly-bounds
      :contains-point? poly-contains-point?
      :centroid poly-centroid
    }
    IRotatable {
      :rotate poly-rotate
    }
    IScalable {
      :scale poly-scale
      :scale-n poly-scale-n
    }
    ITranslatable {
      :translate poly-translate
    }
    ITransformable {
      :transform poly-transform
    }
    IShape2D {
      :area poly-area
      :bounding-circle poly-bounding-circle
      :circumference poly-circumference
      :edges poly-edges
      :->polygon2d identity
    }
    IPolygon2D {
      :add-vertex poly-add-vertex
      :contains-polygon? poly-contains-polygon?
      :flip poly-flip
      :intersect-polygons poly-intersect-polygon
      :vertex-count poly-count
    }
    ))

(extend-polygon2d Polygon2D)
