(ns toxi.geom.octree
  (:use
    [toxi.math.core]
    [toxi.geom.core]
    [toxi.geom.protocols]
    [toxi.geom.aabb]))

(def ^{:dynamic true} *max-depth* 8)

(defn octree-node
  ([parent id]
    (let [bounds (:bounds parent)
          mi (bounds-min bounds) mx (bounds-max bounds)
          [x1 x2] (if (pos? (bit-and id 1)) [(:x bounds) (:x mx)] [(:x mi) (:x bounds)])
          [y1 y2] (if (pos? (bit-and id 2)) [(:y bounds) (:y mx)] [(:y mi) (:y bounds)])
          [z1 z2] (if (pos? (bit-and id 4)) [(:z bounds) (:z mx)] [(:z mi) (:z bounds)])]
      (octree-node (vec3d x1 y1 z1) (vec3d x2 y2 z2) parent)))
  ([mi mx parent]
	  {:bounds (aabb-from-minmax mi mx)
	   :depth (if (nil? parent) 0 (inc (:depth parent)))}))

(defn octant [bounds x y z]
    (bit-or (if (>= x (:x bounds)) 1 0) (if (>= y (:y bounds)) 2 0) (if (>= z (:z bounds)) 4 0)))

(defn leaf? [n] (= (:depth n) *max-depth*))

(defn set-voxel-at [n x y z v]
  (let [id (octant (:bounds n) x y z)
        nodes (if (nil? (:nodes n)) [nil nil nil nil nil nil nil nil] (:nodes n))]
    (if (leaf? n)
      (assoc n :value v)
      (assoc n :nodes
        (assoc nodes id 
           (set-voxel-at (if (nil? (get nodes id))
                        (octree-node n id)
                        (get nodes id)) x y z v))))))

(defn walk [n limit-fn visitor-fn]
  (reduce
    (fn[acc child]
      (if-not (nil? child)
        (if (limit-fn child)
          (reduce (fn[a b] (conj a b)) acc (walk child limit-fn visitor-fn))
          (conj acc (visitor-fn child)))
        acc))
    [] (:nodes n)))
