(ns toxi.geom.line2d
  (:require
    [toxi.math.core :as math])
  (:use
    [toxi.geom.protocols]
    [toxi.geom.core :only [vec2d line2d rect circle polygon2d]]
    [toxi.geom.vec2d])
  (:import
    [toxi.geom.types Line2D]))

(defn- l2d-split
  {:added "0.1"
   :java-id "toxi.geom.Line2D.splitIntoSegments"}
  ([line ^double len] (l2d-split line len [] true))
  ([line ^double len coll] (l2d-split line len coll true))
  ([line len coll addfirst?]
    (let [a (:a line) b (:b line)
          coll2 (if addfirst? (conj coll a) coll)
          dist (distance a b)
          step (limit (sub b a) len)]
      (if (> dist len)
        (loop [pos (add a step)
               acc coll2
               d dist]
          (if (<= d len)
            (conj acc b)
            (recur (add pos step) (conj acc pos) (- d len))))
        (conj coll2 b)))))

(defn- l2d-classify-point
  {:added "0.1"
   :java-id "toxi.geom.Line2D.classifyPoint"}
  [line p]
  (let [a (:a line)
        normal (perpendicular (sub (:b line) a))
        d (dot (sub p a) normal)]
    (Math/signum (double d))))

(defn- l2d-contains-point?
  [line p]
  (zero? (l2d-classify-point line p)))

(defn- l2d-point-on-line
  {:added "0.1"
   :java-id "toxi.geom.Line2D.classifyPoint"}
  [line ^double t] (interpolate (:a line) (:b line) t))

(defn- l2d-random-point
  [line] (l2d-point-on-line line (math/random)))

(defn- l2d-closest-point-to
  {:added "0.1"
   :java-id "toxi.geom.Line2D.closestPointTo"}
  [line p]
  (let [a (:a line)
        b (:b line)
        v (sub b a)
        t (/ (dot (sub p a) v) (mag-squared v))]
    (cond
      (neg? t) a
      (> t 1.0) b
      :else (interpolate a b t))))

(defn- l2d-distance-to-point
  {:added "0.1"
   :java-id "toxi.geom.Line2D.distanceToPoint"}
  [line p] (distance (l2d-closest-point-to line p) p))

(defn- l2d-intersect-line
  {:added "0.1"
   :java-id "toxi.geom.Line2D.intersectLine"}
  [l1 l2]
  (let [ax1 (:x (:a l1)) ay1 (:y (:a l1)) bx1 (:x (:b l1)) by1 (:y (:b l1))
        ax2 (:x (:a l2)) ay2 (:y (:a l2)) bx2 (:x (:b l2)) by2 (:y (:b l2))
        denom (- (* (- by2 ay2) (- bx1 ax1)) (* (- bx2 ax2) (- by1 ay1)))
        na (- (* (- bx2 ax2) (- ay1 ay2)) (* (- by2 ay2) (- ax1 ax2)))
        nb (- (* (- bx1 ax1) (- ay1 ay2)) (* (- by1 ay1) (- ax1 ax2)))]
    (if-not (zero? denom)
      (let [ua (/ na denom) ub (/ nb denom) ipos (point-on-line l1 ua)]
        (if (and (>= ua 0.0) (<= ua 1.0) (>= ub 0.0) (<= ub 1.0))
          {:type :intersect :pos ipos :ua ua :ub ub}
          {:type :no-intersect :pos ipos :ua ua :ub ub}))
      (if (= 0.0 na nb)
        (if (zero? (l2d-distance-to-point l1 (:a l2)))
          {:type :coincident}
          {:type :coincident-no-intersect})
        {:type :parallel}))))

(defn- l2d-direction
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getDirection"}
  [line] (normalize (sub (:b line) (:a line))))

(defn- l2d-midpoint
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getMidPoint"}
  ([line] (l2d-midpoint (:a line) (:b line)))
  ([a b] (interpolate a b 0.5)))

(defn- l2d-offset-and-grow-by
  {:added "0.1"
   :java-id "toxi.geom.Line2D.offsetAndGrowBy"}
  ([line offset] (l2d-offset-and-grow-by line offset 1.0 nil))
  ([line offset scale] (l2d-offset-and-grow-by line offset scale nil))
  ([line offset scale ref]
    (let [m (l2d-midpoint line)
          d (l2d-direction line)
          ds (scale-n d scale)
          t (perpendicular d)
          n (normalize t
              (if (and
                    (not (nil? ref))
                    (neg? (dot (sub m ref) t)))
                (* -1 offset) offset))
          a (add (:a line) n)
          b (add (:b line) n)]
      (assoc line :a (sub a ds) :b (add b ds)))))

(defn- l2d-length
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getLength"}
  [line] (distance (:a line) (:b line)))

(defn- l2d-length-squared
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getLengthSquared"}
  [line] (distance-squared (:a line) (:b line)))

(defn- l2d-scale-length
  {:added "0.1"
   :java-id "toxi.geom.Line2D.scaleLength"}
  [line scale]
  (let [delta (* (- 1 scale) 0.5)
        a (:a line)
        b (:b line)]
    (assoc line :a (interpolate a b delta) :b (interpolate b a delta))))

(defn- l2d-has-endpoint?
  {:added "0.1"
   :java-id "toxi.geom.Line2D.hasEndPoint"}
  [line p] (or (= (:a line) p) (= (:b line) p)))

(defn- l2d-normal
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getNormal"}
  [line] (normalize (sub (:b line) (:a line))))

(defn- l2d-theta
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getTheta"}
  [line] (heading (sub (:b line) (:a line))))

(defn- l2d-bounds
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getBounds"}
  [line] (rect
           (minv (:a line) (:b line))
           (maxv (:a line) (:b line))))

(defn- l2d-bounding-circle
  {:added "0.1"
   :java-id "toxi.geom.Line2D.getBounds"}
  [line] (circle (l2d-midpoint line) (* 0.5 (l2d-length line))))

(defn extend-line2d
  [type]
  (extend type
    IClassifiable {
      :classify-point l2d-classify-point
    }
    IScalable {
      :scale-n l2d-scale-length
    }
    IShape {
      :bounds l2d-bounds
      :centroid l2d-midpoint
      :contains-point? l2d-contains-point?
      :random-point l2d-random-point
    }
    IShape2D {
      :area (fn[l] 0)
      :bounding-circle l2d-bounding-circle
      :circumference (fn[l] (* 2 (l2d-length l)))
      :edges (fn[l] [(identity l)])
      :->polygon2d (fn[l] (polygon2d (:a l) (:b l)))
    }
    ILine {
      :intersect-lines l2d-intersect-line
      :line-closest-point-to l2d-closest-point-to
      :line-direction l2d-direction
      :line-distance-to-point l2d-distance-to-point
      :line-has-endpoint? l2d-has-endpoint?
      :line-length l2d-length
      :line-length-squared l2d-length-squared
      :line-normal l2d-normal
      :offset-and-grow-by l2d-offset-and-grow-by
      :point-on-line l2d-point-on-line
      :split-line l2d-split
    }
    ILine2D {
      :theta l2d-theta
    }))

(extend-line2d Line2D)
