(ns toxi.geom.ellipse
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math])
  (:use
    [toxi.geom.protocols]
    [toxi.geom.core :only [vec2d line2d rect circle ellipse polygon2d]]
    [toxi.geom.vec2d])
  (:import
    [toxi.geom.types Ellipse]))

(def ^{:dynamic true} *ellipse-res* 20)

(defn- ellipse-foci
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getFoci"}
  [e]
  (let [r (:radius e) x (:x e) y (:y e) focus (mag r)]
    (if (> (:x r) (:y r))
      [(vec2d (- x focus) y) (vec2d (+ x focus) y)]
      [(vec2d x (- y focus)) (vec2d x (+ y focus))])))

(defn- ellipse-contains-point?
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.containsPoint"}
  [e p]
  (let [r (:radius e)
        nx (/ (- (:x p) (:x e)) (:x r))
        ny (/ (- (:y p) (:y e)) (:y r))
        nd (+ (* nx nx) (* ny ny))]
    (<= nd 1.0)))

(defn- ellipse-random-point
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getRandomPoint"}
  [e]
  (-> (vec2d-from-theta (math/random math/TWO_PI))
    (scale (scale-n (:radius e) (math/random 1.0)))
    (add e)))

(defn- ellipse-area
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getArea"}
  [e] (let[r (:radius e)] (* math/PI (:x r) (:y r))))

(defn- ellipse-circumference
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getCircumference"}
  [e] (* math/PI (Math/sqrt (* 0.5 (mag-squared (:radius e))))))

(defn- ellipse-bounds
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getBounds"}
  [e] (rect (sub e (:radius e)) (add e (:radius e))))

(defn- ellipse-bounding-circle
  [e] (circle (:x e) (:y e) (max (:x (:radius e)) (:y (:radius e)))))

(defn- ellipse->polygon2d
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.toPolygon2D"}
  ([e] (ellipse->polygon2d e *ellipse-res*))
  ([e res]
  (let [step (/ math/TWO_PI res) r (:radius e)]
    (polygon2d (reduce (fn[vertices i]
              (conj vertices
                (add (scale (vec2d-from-theta (* i step)) r) e)))
      [] (range res))))))

(defn- ellipse-edges
  ([e] (edges (ellipse->polygon2d e *ellipse-res*)))
  ([e res] (edges (ellipse->polygon2d e res))))

(defn extend-ellipse
  [type]
  (extend-vec2d type)
  (extend type
    IShape {
      :bounds ellipse-bounds
      :contains-point? ellipse-contains-point?
      :random-point ellipse-random-point
    }
    IShape2D {
      :area ellipse-area
      :bounding-circle ellipse-bounding-circle
      :circumference ellipse-circumference
      :edges ellipse-edges
      :->polygon2d ellipse->polygon2d
    }
    IEllipse {
      :foci ellipse-foci
    }))

(extend-ellipse Ellipse)