(ns toxi.geom.core
  (:require
    [toxi.data.index :as index])
  (:use
    [toxi.core]
    [toxi.math.core]
    [toxi.geom.protocols]
    [toxi.geom.types])
  (:import
    [toxi.geom.types
     Vec2D Vec3D Vec4D
     Circle Ellipse
     Line2D Triangle2D Rect Polygon2D
     Matrix4x4 Quaternion
     AABB Sphere
     TriangleMesh3D Face3D]))

(defn vec2d
  "Creates a new Vec2D instance. Coordinates are cast to Double.
  If the single argument is a collection, the first two items are used as x & y.
  If the single argument is a number, it's used for both XY: (vec2d 1) = (vec2d 1 1).
  If no arguments are specified xy are set to 0.0."
  {:added "0.1"
   :java-id "toxi.geom.Vec2D"}
  ([] (Vec2D. 0.0 0.0))
  ([v]
    (cond
      (map? v) (Vec2D. (:x v) (:y v))
      (coll? v) (Vec2D. (first v) (second v))
      (number? v) (Vec2D. v v)))
  ([^double x ^double y] (Vec2D. (double x) (double y))))

(defn vec3d
  "Creates a new Vec3D instance. Coordinates are cast to Double.
If the single argument is a collection, the first three items are used as x, y & z.
If the single argument is a number, it's used for xyz: (vec3d 1) = (vec3d 1 1 1).
If no arguments are specified, xyz are set to 0.0."
  {:added "0.1"
   :java-id "toxi.geom.Vec3D"}
  ([] (Vec3D. 0.0 0.0 0.0))
  ([v]
    (cond
      (map? v) (Vec3D. (:x v) (:y v) (:z v))
      (coll? v) (Vec3D. (first v) (second v) (nth v 2))
      (number? v) (Vec3D. v v v)))
  ([^double x ^double y ^double z] (Vec3D. (double x) (double y) (double z))))

(defn vec4d
  "Creates a new Vec4D instance. Coordinates are cast to Double.
If argument is a collection, the first four items are used as x, y, z & w.
If no arguments are specified xyz are set to 0.0 and w to 1.0."
  {:added "0.1"
   :java-id "toxi.geom.Vec4D"}
  ([] (Vec4D. 0.0 0.0 0.0 1.0))
  ([coll] (Vec4D. (first coll) (second coll) (nth coll 2) (nth coll 3)))
  ([xyz ^double w] (Vec4D. (:x xyz) (:y xyz) (:z xyz) w))
  ([^double x ^double y ^double z ^double w]
    (Vec4D. (double x) (double y) (double z) (double w))))

(defn line2d
  {:added "0.1"
   :java-id "toxi.geom.Line2D"}
  ([a b] (Line2D. a b))
  ([^double x1 ^double y1 ^double x2 ^double y2] (line2d (vec2d x1 y1) (vec2d x2 y2))))

(defn triangle2d
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D"}
  ([a b c] (Triangle2D. a b c))
  ([x1 y1 x2 y2 x3 y3]
    (triangle2d (vec2d x1 y1) (vec2d x2 y2) (vec2d x3 y3))))

(defn rect
  {:added "0.1"
   :java-id "toxi.geom.Rect"}
  ([] (Rect. 0 0 0 0))
  ([^Vec2D min ^Vec2D max]
    (let [w (- (:x max) (:x min)) h (- (:y max) (:y min))]
      (Rect. (:x min) (:y min) w h)))
  ([^double x ^double y ^double w ^double h] (Rect. x y w h)))

(defn polygon2d
  ([] (Polygon2D. []))
  ([coll] (Polygon2D. (vec (map vec2d coll))))
  ([v & more] (Polygon2D. (vec (map vec2d (concat [v] more))))))

(defn circle
  {:added "0.1"
   :java-id "toxi.geom.Circle"}
  ([] (Circle. 0.0 0.0 1.0))
  ([radius] (Circle. 0 0 (abs radius)))
  ([origin ^double radius] (Circle. (:x origin) (:y origin) (abs radius)))
  ([^double x ^double y ^double radius] (Circle. x y (abs radius))))

(defn ellipse
  {:added "0.1"
   :java-id "toxi.geom.Ellipse"}
  ([] (Ellipse. 0.0 0.0 (vec2d 1.0)))
  ([origin radii]
    (cond
      (number? radii)
      (Ellipse. (:x origin) (:y origin) (vec2d (abs radii)))
      (map? radii)
      (Ellipse. (:x origin) (:y origin) (vec2d (abs (:x radii)) (abs (:y radii))))))
  ([origin ^double rx ^double ry]
    (Ellipse. (:x origin) (:y origin) (vec2d (abs rx) (abs ry))))
  ([^double x ^double y ^double rx ^double ry]
    (Ellipse. x y (vec2d (abs rx) (abs ry)))))

(defn triangle-mesh3d
  ([] (TriangleMesh3D.
        (index/make-index) ; vertex index
        (index/make-index) ; normal index
        [] ; vertex normal IDs
        {} ; attrib index
        []))) ; face list

(defn face3d
  ([^Integer a ^Integer b ^Integer c]
    (Face3D. (int a) (int b) (int c) (int -1) {}))
  ([^Integer a ^Integer b ^Integer c attribs]
    (Face3D. (int a) (int b) (int c) (int -1) attribs)))

(defn sphere
  ([^double radius]
    (Sphere. 0 0 0 radius))
  ([v ^double radius]
    (Sphere. (:x v) (:y v) (:z v) radius))
  ([x y z radius]
    (Sphere. x y z radius)))

(defn matrix4x4
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4"}
  ([]
    (Matrix4x4.
      1 0 0 0
      0 1 0 0
      0 0 1 0
      0 0 0 1))
  ([mat]
    (Matrix4x4.
      (get mat 0) (get mat 1) (get mat 2) (get mat 3)
      (get mat 4) (get mat 5) (get mat 6) (get mat 7)
      (get mat 8) (get mat 9) (get mat 10) (get mat 11)
      (get mat 12) (get mat 13) (get mat 14) (get mat 15)))
  ([m00 m01 m02 m03 m10 m11 m12 m13 m20 m21 m22 m23 m30 m31 m32 m33]
    (Matrix4x4.
      m00 m01 m02 m03
      m10 m11 m12 m13
      m20 m21 m22 m23
      m30 m31 m32 m33)))

(defn aabb
  {:added "0.1"
   :java-id "toxi.geom.AABB"}
  ([] (AABB. 0 0 0 (vec3d 0.5 0.5 0.5)))
  ([extent] (AABB. 0 0 0 (vec3d extent)))
  ([origin extent] (AABB. (:x origin) (:y origin) (:z origin) extent))
  ([x y z ex ey ez] (AABB. x y z (vec3d ex ey ez))))

(defn quaternion
  {:added "0.1"
   :java-id "toxi.geom.Quaternion"}
  ([] (Vec4D. 0.0 0.0 0.0 1.0))
  ([^double w xyz] (Vec4D. (:x xyz) (:y xyz) (:z xyz) w))
  ([^double w ^double x ^double y ^double z] (Vec4D. x y z w)))
