(ns ^:figwheel-hooks cljs-map-pins.core
  "The cljs-map-pins core namespace."
  (:require [goog.dom :as gdom]
            [schema.core :as s :include-macros true]
            ["ol" :as ol :refer (Map View Feature)]
            ["ol/control" :as olc :refer (Control Attribution)]
            ["ol/geom" :as olg :refer (Point)]
            ["ol/style" :as ols :refer (Style Icon)]
            ["ol/layer" :as oll :refer (Tile Vector)]
            ["ol/source" :as olsrc :refer (OSM Vector)]))

;; Utility

(defn- get-map-element-id
  "If the argument is a HTML element returns the id attribute, otherwise if
  string returns the argument."
  {:added "0.1.0"}
  [map-element]
  (if-not (string? map-element)
    (.-id map-element)
    map-element))

(defn- map->js
  "Converts all values of the map to JS object alongside the map
  itself."
  {:added "0.1.0"}
  [m]
  (clj->js (zipmap (keys m) (map clj->js (vals m)))))

(defprotocol FeatureHandler
  "Protocol describing the behavior of a map feature on-click handler."
  (handle [this feature]))


;; Validation

(s/defschema OLOpts
  "A Schema describing the OpenLayers options map used for map drawing.
This is direct copy from here: https://openlayers.org/en/latest/apidoc/module-ol_View-View.html"
  {(s/required-key :center) [s/Num]
   (s/optional-key :extent) [s/Num]
   (s/required-key :zoom) s/Int
   (s/required-key :projection) s/Str
   (s/optional-key :minZoom) s/Int
   (s/optional-key :maxZoom) s/Int
   (s/optional-key :controls) [s/Any]
   (s/optional-key :withoutAttribution) s/Bool
   (s/optional-key :constrainRotation) s/Bool
   (s/optional-key :enableRotation) s/Bool
   (s/optional-key :constrainOnlyCenter) s/Bool
   (s/optional-key :smoothExtentConstraint) s/Bool
   (s/optional-key :maxResolution) s/Num
   (s/optional-key :minResolution) s/Num
   (s/optional-key :multiWorld) s/Num
   (s/optional-key :constrainResolution) s/Bool
   (s/optional-key :smoothResolutionConstraint) s/Bool
   (s/optional-key :showFullExtent) s/Bool
   (s/optional-key :resolution) s/Num
   (s/optional-key :resolutions) [s/Num]
   (s/optional-key :rotation) s/Num
   (s/optional-key :zoomFactor) s/Num
   (s/optional-key :padding) [s/Num]})

(defn ol-options-valid?
  "A function asserting the validity of OpenLayers options against OLOpts Schema."
  {:added "0.1.0"}
  [ol-options]
  (s/validate OLOpts ol-options))

(s/defschema Features
  "A Schema describing the map of features that will be drawn on the map."
  [{(s/required-key :name) s/Str
    (s/required-key :type) s/Keyword
    (s/required-key :title) s/Str
    (s/required-key :description) s/Str
    (s/optional-key :url) s/Str
    (s/required-key :coordinates) [s/Num]
    (s/optional-key :img) s/Str
    (s/optional-key :style) s/Any}])

(defn features-valid?
  "A function asserting the validity of map features map against the Features Schema."
  {:added "0.1.0"}
  [features]
  (s/validate Features features))

(s/defschema Handlers
  "A Schema describing the map of map feature handlers. There can be as many
handlers as there are feature types with additional two handlers: one default
and one used when there are no features in the on-click event."
  {s/Keyword (s/protocol FeatureHandler)
   :default (s/protocol FeatureHandler)
   :no-feature s/Any})

(defn handlers-valid?
  "A function validating the feature handlers map against the Handlers Schema."
  {:added "0.1.0"}
  [handlers]
  (s/validate Handlers handlers))


;; Features

(defn get-feature-name
  "Returns the :name value from a feature."
  {:added "0.2.0"}
  [f]
  (.get f "name"))

(defn get-feature-title
  "Returns the :title value from a feature."
  {:added "0.2.0"}
  [f]
  (.get f "title"))

(defn get-feature-description
  "Returns the :description value from a feature."
  {:added "0.2.0"}
  [f]
  (.get f "description"))

(defn get-feature-url
  "Returns the :url value from a feature."
  {:added "0.2.0"}
  [f]
  (.get f "url"))

(defn create-feature
  "Creates a ol.Feature instance from the feature map. It assigns the ol.geom.Point
  instance with the feature coordinates and ol.style.Style instance with the
  ol.style.Icon for the given feature image path."
  {:added "0.1.0"}
  [{coords :coordinates img :img style :style :as feature}]
  (let [f (ol/Feature. (clj->js feature))
        g (olg/Point. (clj->js coords))
        s (cond
            img (ols/Style. #js {:image (ols/Icon. #js {:src img})})
            :else style)]
    (doto f
      (.setGeometry g)
      (.setStyle s))))


;; Controls

(defn contains-attribution-control?
  "Asserts of a collection contains an instance of ol.control.Attribution class."
  {:added "0.2.0"}
  [cs]
  (some #(instance? olc/Attribution %) cs))

(defn add-controls!
  "Adds passed OpenLayers ol.control.Control instances to the given map."
  {:added "0.1.0"}
  [^js/ol.Map m {no-attr :withoutAttribution controls :controls}]
  (let [cs (if controls
             (if (or (contains-attribution-control? controls)
                     no-attr)
               controls
               (conj controls (olc/Attribution.)))
             (if no-attr
               []
               [(olc/Attribution.)]))]
    (doseq [c cs]
      (.addControl m c)))
  m)

(defn map-on-feature-pointer-handler!
  "Creates an event handler and assigns it to the map.
  It will change the pointer style to 'hand'
  when mouse crosses over a feature on the map."
  {:added "0.1.0"}
  [^js/ol.Map m map-element]
  (doto m
    (.on "pointermove" (fn [^js/Event e] (when-not (.-dragging e)
                                           (let [^js/Event original-event (.-originalEvent e)
                                                 ^js/ol.Pixel pixel (.getEventPixel m original-event)]
                                             (if (.hasFeatureAtPixel m pixel)
                                               (gdom/setProperties map-element #js {:style "cursor: pointer"})
                                               (gdom/setProperties map-element #js {:style "cursor: ''"}))))))))

(defn feature-event-handler
  "Returns an event-handling function that will be called by OpenLayers
  when a feature is clicked on the map. The function will retrieve a
  FeatureHandler instance that is mapped to the feature type keyword
  and execute its handling function. If there are no feature for the type
  executes the default handler."
  {:added "0.1.0"}
  [handlers]
  (fn [feature]
    (let [type (keyword (.get feature "type"))
          handler (type handlers)
          default-handler (:default handlers)]
      (if-not (nil? handler)
        (handle handler feature)
        (handle default-handler feature)))))

(defn map-on-click-handler!
  "Creates an on-click event handler and assigns it to the map.
  Calls the feature-event-handler function when there is a feature on
  the clicked pixel otherwise executes the no-op handler supplied in the
  handlers map."
  {:added "0.1.0"}
  [^js/ol.Map map {noop :no-feature :as handlers}]
  (.on map "singleclick" (fn [^js/Event e]
                           (let [^js/Event original-event (.-originalEvent e)
                                 ^js/ol.Pixel pixel (.getEventPixel map original-event)]
                             (if (.hasFeatureAtPixel map pixel)
                               (.forEachFeatureAtPixel map (.-pixel e) (feature-event-handler handlers))
                               (noop))))))


;; Main

(defn draw-pins!
  "Draws a map in the passed HTML element. The element must have non-zero
  width and height for the map to show. Takes a map of Open Layers options
  and passes it to the ol.Map instance. Draws all the features from the features
  vector and assigns the appropriate handlers from the handlers map.
  Throws an exception if any of the parameters don't match their respective
  Schema."
  {:added "0.1.0"}
  [map-element ol-options features handlers]
  (when (and
         (not (nil? map-element))
         (ol-options-valid? ol-options)
         (features-valid? features)
         (handlers-valid? handlers))
    (let [source (olsrc/OSM. #js {:layer "sat"})
          layer (oll/Tile. #js {:source source})
          vector-source (olsrc/Vector. #js {:features (clj->js (map create-feature features))})
          vector-layer (oll/Vector. #js {:source vector-source})
          view (ol/View. (map->js ol-options))]
      (doto (ol/Map. #js {:layers #js [layer vector-layer]
                          :target (get-map-element-id map-element)
                          :view view
                          :controls #js []})
        (add-controls! ol-options)
        (map-on-feature-pointer-handler! map-element)
        (map-on-click-handler! handlers)))))
