# clj-pnm

![CI](https://github.com/28/clj-pnm/actions/workflows/CI.yml/badge.svg "CI")
[![Clojars Project](https://img.shields.io/clojars/v/org.theparanoidtimes/clj-pnm.svg)](https://clojars.org/org.theparanoidtimes/clj-pnm)

A Clojure(Script) library for reading and writing Netpbm format.
Supports plain-test formats: *p1* (*.pbm*), *p2* (*.pgm*) and *p3* (*.ppm*).

Notes on the state of the library:

- Plain-text Netpbm is supported and implementation is stable.
- The Netpbm format completely ignores whitespace, however *clj-pnm* requires that characters are separated by whitespace.
- Binary formats (*p4*, *p5* and *p6*) are not supported.
- [*pam*](http://netpbm.sourceforge.net/doc/pam.html) format is not supported.
- Writing in-line comments to a file (for example in the header part) is not supported.

Good explanation of the format and the resources that are used as guides
for this library can be found [here](http://netpbm.sourceforge.net/doc/index.html)
and [here](https://en.wikipedia.org/wiki/Netpbm_format).

## Usage

The library is available on *Clojars*. Add it to your project as a dependency.

`deps.edn`:

``` clojure
org.theparanoidtimes/clj-pnm {:mvn/version "0.2.0"}
```

`project.clj`:

``` clojure
[org.theparanoidtimes/clj-pnm "0.2.0"]
```

`require` the library:

``` clojure
(require '[clj-pnm.core :as pnm])
=> nil
```

Read the *pbm* format:

``` clojure
(def pbm
"P1
 2
 2
 1 1
 1 1")
=> #'user/pbm

(pnm/read-pnm pbm)
=> {:type :p1, :width 2, :height 2, :map [[1 1] [1 1]]}
```

Read the *pgm* format:

```clojure
(def pgm
"P2
 3
 2
 4
 # pgm-comment
 1 2 3
 4 3 2")
=> #'user/pgm

(pnm/read-pnm pgm)
=> {:type :p2, :width 3, :height 2, :max-value 4, :map [[1 2 3] [4 3 2]] :comments ("pgm-comment")}
```

Read the *ppm* format:

```clojure
(def ppm
"P3
 2
 2
 255
 # ppm-comment
 255 0 255 128 52 123
 0 0 0 45 45 45")
=> #'user/ppm

(pnm/read-pnm ppm)
=> {:type :p3
    :width 2
    :height 2
    :max-value 255
    :map [[(255 0 255) (128 52 123)]
          [(0 0 0) (45 45 45)]]
    :comments ("ppm-comment")} ; prettified
```

You can extract comments separately if necessary:

```clojure
(def pbm-with-comments
"P1
 1
 1
 # A comment
 # Another comment
 1")
=> #'user/pbm-with-comments

(pnm/get-comments (pnm/read-lines pbm-with-comments))
=> ("A comment" "Another comment")
```

Some files have in-line comments in the header part.
They are handled in the same way:

``` clojure
(def pbm-with-in-line-comments
"P1
 1 # this is the width
 1
 # A comment
 # Another comment
 1")
=> #'user/pbm-with-in-line-comments

(pnm/read-pnm pbm-with-in-line-comments)
=> {:type :p1, :width 1, :height 1, :map [[1]] :comments ("1  this is the width" "A comment" "Another comment")}
```

``` clojure
(def pbm-with-in-line-comments
"P1
 1
 1 # this is the height
 # A comment
 # Another comment
 1")
=> #'user/pbm-with-in-line-comments

(pnm/get-comments (pnm/read-lines pbm-with-in-line-comments))
=> ("1  this is the width" "A comment" "Another comment")
```

Clojure version of *clj-pnm* can work with files. To read a *pnm* file
pass it to the `read-pnm` function:

``` clojure
(require '[clojure.java.io :as io])
=> nil

(slurp (io/file "test.pgm"))
=> P2
 3
 2
 4
 1 2 3
 4 5 6 ; prettified

(pnm/read-pnm (io/file "test.pgm"))
=> {:type :p2, :width 3, :height 2, :max-value 4, :map [[1 2 3] [4 5 6]]}
```

Writing a file is just a matter of passing a map like the ones explained
above. It can include comments which are always written after the header part.

``` clojure
(pnm/write-pnm {:type :p1 :width 1 :height 1 :map [[1]]} "out.pbm")
=> ...out.pbm file...

(slurp *1)
=> "P1
 1
 1
 1" ;prettified

(pnm/write-pnm
{:type :p3 :width 1 :height 1 :max-value 255 :map [[1]] :comments '("A comment" "Another comment")}
"out.ppm")
=> ...out.ppm file...

(slurp *1)
=> "P3
 1
 1
 255
 # A comment
 # Another comment
 255 255 255" ; prettified
```

## License

Copyright 2018-2023, Dejan Josifović, The Paranoid Times

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
