(ns org.tcrawley.datastar-pedestal-adapter.impl
  (:require
    [clojure.core.async :as a]
    [clojure.string :as string]
    [starfederation.datastar.clojure.adapter.common :as ac]
    [starfederation.datastar.clojure.protocols :as p]
    [starfederation.datastar.clojure.utils :as u])
  (:import
    (java.io
      #_ByteArrayOutputStream
      Closeable)
    (java.util.concurrent.locks
      ReentrantLock)))

(def basic-profile
  "Basic write profile using temporary [[StringBuilder]]s, no output stream and
  no compression."
  {ac/write! (fn write!* [event-type data-lines {:d*.sse/keys [id]}]
               (cond-> {:name event-type
                        :data (string/join "\n" data-lines)}
                 id (assoc :id id)))})

;; -----------------------------------------------------------------------------
;; Sending events machinery
;; -----------------------------------------------------------------------------
(defn- ->send-simple
  [ch write-profile]
  (let [write! (ac/write! write-profile)]
    (fn
      ([])
      ([event-type data-lines opts]
       (let [event (write! event-type data-lines opts)]
         (a/>!! ch event))))))

#_(defn- flush-baos!
  [^ByteArrayOutputStream baos ch]
  (let [msg (.toByteArray baos)]
    (.reset baos)
    (a/>!! ch msg)))

#_(defn- ->send-with-output-stream
  [ch write-profile]
  (let [^ByteArrayOutputStream baos (ByteArrayOutputStream.)
        {wrap-os ac/wrap-output-stream
         write! ac/write!} write-profile
        writer (wrap-os baos)]
    (fn
      ([]
       ;; Close the writer first to finish the gzip process
       (.close ^Closeable writer)
       ;; Flush towards SSE out
       (flush-baos! baos ch))
      ([event-type data-lines opts]
       (write! writer event-type data-lines opts)
       (ac/flush writer)
       (flush-baos! baos ch)))))

(defn ->send!
  [ch opts]
  (let [write-profile (or (ac/write-profile opts)
                          ac/basic-profile)]
    (if (ac/wrap-output-stream write-profile)
      (throw (UnsupportedOperationException. "Sending with an output stream is unsupported in the pedestal adapter."))
      #_(->send-with-output-stream ch write-profile)
      (->send-simple ch write-profile))))

(defn- call-on-close
  [this reason]
  (when-some [on-close (.on-close this)]
    (u/lock! (.lock this) (on-close this reason))))

(defn ->on-client-disconnect
  [sse-gen-promise]
  (fn [_pedestal-context]
    (when (realized? sse-gen-promise)
      (call-on-close @sse-gen-promise :client-disconnect))))

;; -----------------------------------------------------------------------------
;; SSE gen
;; -----------------------------------------------------------------------------
(deftype SSEGenerator
    [ch lock send! on-close on-exception]
    p/SSEGenerator
    (send-event!
      [this event-type data-lines opts]
      (u/lock! lock
        (try
          (send! event-type data-lines opts)
          (catch Exception e
            (when (on-exception this e {:sse-gen this
                                        :event-type event-type
                                        :data-lines data-lines
                                        :opts opts})
              (p/close-sse! this))
            false))))
    (get-lock [_] lock)
    (close-sse!
      [this]
      (a/close! ch)
      (call-on-close this :server-close))
    (sse-gen? [_] true)

    Closeable
    (close
      [this]
      (p/close-sse! this)))

(defn ->sse-gen
  ([ch send!]
   (->sse-gen ch send! {}))
  ([ch send! opts]
   (SSEGenerator. ch (ReentrantLock.) send!
                  (ac/on-close opts)
                  (or (ac/on-exception opts)
                      ac/default-on-exception))))

