(ns taoclj.junction.routing
  (:require [taoclj.junction.path :as path]))


(defn method-to-handler-type [method]
  (cond (= method :websocket) :websocket
        (some #(= method %) [:get :post]) :http
        :default (throw (Exception. "Unknown handler method!"))))


(defn match-route
  "Determines if a single route is matched, if no match returns nil."
  [route method path]
  (if (some #(= method %) [:get :post :websocket]) ;; is the method allowed?
    (if-let [matched-path-params (path/match (first route) path)]
      (if-let [matched-handler (-> route second method)]
        {:handler [(method-to-handler-type method) matched-handler]
         :params matched-path-params
         :authorize (-> route second :authorize)}))))



(defn match-routes
  "Returns the first matched route if any in a collection of routes"
  [routes method path]
  (first (filter #(not (nil? %))
                 (map #(match-route % method path)
                      routes))))



;; Route Grouping Functions

(defn add-route-option
  "Adds value to a route's options map."
  [route key val]
  [(first route)
   (assoc (second route) key val)])

(defn flatten-route-group [route-group]
  (let [type (first route-group)]
    (cond (= :public type) (map #(add-route-option % :authorize :public)
                                (drop 1 route-group))
          (= :authorize type) (map #(add-route-option % :authorize (second route-group))
                                   (drop 2 route-group))
          :default (throw
                    (Exception. "Route group types must be either :public or :authorize")))))

(defn flatten-route-groups
  "Flattens an auth grouped collection of routes, placing the group setting in options map.
This is done to make it simple to search the routing table. Returns single sequence of routes"
  [routes]
  (apply concat (map #(flatten-route-group %) routes)))






