(ns taoclj.junction
  (:require [taoclj.junction.routing :as routing]))


(defn in?
  "Does an element exists in a sequence?"
  [sequence element] ;; potentially change the order of these
  (some #(= element %) sequence))

(defn any-matches?
  "Are there any matching elements between sequences?"
  [sequence1 sequence2]
  (some #(in? sequence1 %) sequence2))


(defn route-not-found?
  "Checks to see if result from routing match is not found."
  [match]
  (nil? match))

(defn request-authorized?
  "Checks to see if a user is authorized to to invoke a matched route."
  [match user]
  (if (= (match :authorize) :public) true
      (cond (nil? user) false
            (any-matches? (match :authorize) (user :roles)) true
            :default false)))


(defn merge-params [ctx match]
  (if (nil? match)
    (ctx :params)
     (merge (ctx :params) (match :params))))


(def routes '())
(def default-content-type "text/plain")
(def unmatched-handler (fn [ctx] [404 {:content-type "text/plain"} "not found"]))
(def not-authorized-handler (fn [ctx] [403 {:content-type "text/plain"} "forbidden"]))


(defn dispatch [ctx]
  (let [match (routing/match-routes routes (ctx :request-method) (ctx :uri))
        user (ctx :user)
        handler (cond (route-not-found? match) [:http unmatched-handler]
                        (not (request-authorized? match user)) [:http not-authorized-handler]
                        :default (match :handler))]
    {:handler handler
     :ctx (merge ctx {:params (merge-params ctx match)})}))



(defn set-location "Sets a Location key in a map."
  [headers location]
  (if-not location headers
    (assoc headers "Location" location)))

(defn set-content-type "Sets a Content-Type key in a map."
  [headers default content-type]
  (assoc headers "Content-Type" (if-not content-type default
                                        content-type)))


(defn proxy-ring-response "converts tao response array to ring map"
  [response]
  (let [headers (second response)]
    {:status (first response)
     :headers (-> {}
                  (set-content-type default-content-type (headers :content-type))
                  (set-location (headers :location)))
     :cookies (headers :cookies)
     :body (nth response 2)}))


(defn- set-option! [var-to-alter val]
  (if val (alter-var-root var-to-alter (fn [f] val))))

(defn init [opts]
  (set-option! (var routes) (routing/flatten-route-groups (opts :routes)))
  (set-option! (var unmatched-handler) (opts :unmatched-handler))
  (set-option! (var not-authorized-handler) (opts :not-authorized-handler))
  (set-option! (var default-content-type) (opts :default-content-type)))
