# Foundation/PG

A complete toolkit for talking to Postgres. A work in progress, but usable.




## Key Features & Goals
- Simple to run query sets and transaction sets.
- SQL templating similar to yesql
- Parameters passed as PreparedStatement parameters
- A way to add dynamic query sections
- Automatic conversion of datetime datatypes
- Integer and text array support
- JSON support (partial/pending)
- Multiple result set support
- Automatic conversion from dash to underscore and back
- Keyword case unchanged, rather than automatically converted to lower case.
- Connection pooling with the excellent [HikariCP](http://brettwooldridge.github.io/HikariCP/)



## Comparison & Why use Foundation instead of java.jdbc/yesql/korma/honeysql?


Foundation looks to take the best attributues from other clojure sql libs,
as well as some ideas of it's own and tries to  put them together in a
single consistent easy to use package for the common usages.
For instance, template queries from yesql are great
for complex select queries, but don't work nearly as well for simple selects and inserts/updates
where simple dsl such as JDBC works just fine.

Foundation makes executing multiple queries simpler with a threading syntax, and allows passing the DB as a parameter style.

Using the qry-> and trx-> query macros, you don't need to worry about opening and closing
database connections, it's automatically handled for you. You also have access to the result-sets of prior queries if you need to insert say child records.


Foundation discourages use of SQL DDL such as creating or dropping tables within applications.



## Compared to java.jdbc

Datetimes conversions are handled for you, just supply a java.time.Instant. No more manual
conversion to java.sql.DateTime and back on every query.

Foundation by default converts underscores in the database to dashes in keywords.
Foundation also does not lower case your column names in keywords.

Foundation only supports postgresql, and so can pre-map many of the common
datatypes such as datetimes/json/arrays out of the box.

Foundation can read multiple result sets on queries that return them.

Foundation in general just returns the generated key, or row count effected
by a query, not an entire map of all columns from queries.



## Compared to Yesql

Foundation has a solution for dynamic queries, when you need them.

Foundation discourages use of templated queries for common inserts/updates, and instead suggests
you use the DSL. For complex inserts/updates you can still fall back to templates.

Foundation only supports a single query per file, yesql allows many.

Foundation is aware of postgresql array syntax and does not treat arrays
in queries as parameter placeholders.




## Comparison / Why Use Foundation?
- Automatic conversion from dash to underscore and back.
- Foundation does not change the casing of column names.

- Foundation automatically convert date/times to java.time.Instants.

- Foundation has a way to deal with dynamic queries.

- Foundation converts vectors of strings and vector of ints to array column types.
- Foundation automatically converts maps to json colum type.
- Foundation supports multiple result sets, java.jdbc does not.


## Where does Foundation not do well (yet)
- Lazy queries and/or large result sets
- Full jsonb column type support.
- SSL connection support
- It's still a very young unproven library.
- manually manage transaction lifetime so that actions may be taken within transactions.



## Installation

Add this to your [Leiningen](https://github.com/technomancy/leiningen) `:dependencies`

[![Clojars Project](http://clojars.org/org.taoclj/foundation/latest-version.svg)](http://clojars.org/org.taoclj/foundation)



## Quick Introduction
```clojure

(require '[taoclj.foundation :as pg])


;; for the sql structure see /resources/examples.sql


; define the datasource with your database details
(pg/def-datasource examples-db {:host     "localhost"
                                :port     5432
                                :database "examples_db"
                                :username "examples_app"
                                :password "password"
                                :pooled   false })



; insert a single record, returns the database generated key
(pg/trx-> examples-db
          (pg/insert :categories {:name "Category A"}))

=> 1


; select a single record, returns a map representing the row
(pg/qry-> examples-db
          (pg/select1 :categories {:id 1}))

=> {:id 1, :name "Category A"}


; insert a multiple records, returns the generated keys as sequence
(pg/trx-> examples-db
          (pg/insert :categories [{:name "Category B"}
                                  {:name "Category C"}]))
=> (2 3)


; select multiple records
(pg/qry-> examples-db
          (pg/select :categories {}))

=> ({:id 1, :name "Category A"}
    {:id 2, :name "Category B"}
    {:id 3, :name "Category C"})



; insert a new category and 2 child products
(pg/trx-> examples-db
          (pg/insert :categories {:name "Category D"})
          (pg/insert :products (pg/with-rs

                                 ; a sequence of values for insertion
                                 ["Product D1" "Product D2"]

                                 ; this is the template to use for each item upon insert
                                 ; rs   - implicitly available and is the resultset
                                 ; item - implicitly available name for each value
                                 {:category-id (first rs)
                                  :name item}

                                 )))

=> [4 (1 2)]

```

```sql
-- write more complex queries in sql template files

select p.id, p.name, c.name as category_name
  from products p
    inner join categories c on p.category_id = c.id
  where p.category_id = :category-id
```

```clojure

; define the query and reference the template file.
(pg/def-query my-query
    {:file "path_to/my_query.sql"})

; now use our template query
(pg/qry-> examples-db
          (my-query {:category-id 4}))


=> ({:id 1, :name "Product D1", :category-name "Category D"}
    {:id 2, :name "Product D2", :category-name "Category D"})


```




## Docs and Howto

- [Query Threading - running sets of queries](docs/query-threading.md)
- [Templated Queries - for more complex queries](docs/templated-queries.md)
- [Dynamic Queries - create dynamic sections](docs/dynamic-queries.md)
- [Using JSON - how to use JSON datatypes](docs/json-support.md)
- [Selects - DSL for simple select queries](docs/selecting-data.md)
- [Inserts - DSL for inserting data](docs/inserting-data.md)
- [Updates - DSL for updating data](docs/updating-data.md)
- [Deletes - DSL for deleting data](docs/deleting-data.md)
- [Raw Queries - using raw unsecure sql statements](docs/raw-queries.md)
- [Listen/Notify - push notifications from postgres](docs/listen-notify.md)




## Rationale

I simply wanted something as easy to get data into and out of postgres. I wanted DateTime's converted automatically, and underscores converted to dashes. Also I really wanted a clean sytax structure for exectuting multiple queries at once, such as adding a parent and multiple child records. I wanted sensible result back based on outcome of query. None of the other clojure sql access libaries was quite what I desired.

After much experimentation, I concluded that insert, updates and very simple select statements are better handled using a lightweight DSL. This allow transparent handling of both single items as well as sequences of items, batch inserts, cuts down on number of templated queries we need to write and means we don't have to rely on function naming conventions for return values. For any query beyond the most trivial, it's a better solution to then use a templated query.

The primary goal is ease of use while also encouraging a correct path. I want to embrace postgres to fullest extent possible, and support postgresql extended datatypes (eg arrays, json, hstore, gis)

Why support only postgresql? Simply I've chosen to build first class support for 1 database rather than lowest common denominator support for all given limited time. Also a desire to out of the box map more complex types such as arrays & json, and also to support listen/notify.





## Intentionally Not Included

Support for other databases such as MySql, MS Sql, Oracle, etc.

Data definition sql and schema migrations. We believe that these concerns should
be handled outside of the data layer of your application and no plans to include
these into this library.





## License

Copyright © 2015 Michael Ball

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
