;
; Copyright © 2017 Symphony Software Foundation
; SPDX-License-Identifier: Apache-2.0
;
; Licensed under the Apache License, Version 2.0 (the "License");
; you may not use this file except in compliance with the License.
; You may obtain a copy of the License at
;
;     http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.
;

(ns clj-symphony.message
  "Operations related to messages.  Currently, Symphony supports these message formats:
  1. plain text
  2. messageML, which supports a small number of formatting tags
  3. MessageMLv2 (as of Symphony v1.46)"
  (:require [clojure.string      :as s]
            [clj-symphony.user   :as syu]
            [clj-symphony.stream :as sys]))


(defn msgobj->map
  "Converts a SymMessage object into a map."
  [^org.symphonyoss.symphony.clients.model.SymMessage m]
  (if m
    {
      :message-id     (.getId          m)
      :timestamp      (java.util.Date. (Long/valueOf (.getTimestamp m)))
      :stream-id      (.getStreamId    m)
      :user-id        (.getFromUserId  m)
      :type           (.getMessageType m)   ; This seems to be null or blank most of the time...
      :text           (.getMessage     m)
      :attachment     (.getAttachment  m)
      :entity-data    (.getEntityData  m)
    }))


(defn escape
  "Escapes the given string for MessageML."
  [^String s]
  (when s
    (org.apache.commons.lang3.StringEscapeUtils/escapeXml11 s)))


(defn- ^org.symphonyoss.symphony.clients.model.SymMessage build-sym-message
  "Builds a SymMessage object from the given message string and entity data (optional)."
  [^String m ^String ed]
  (doto
    (org.symphonyoss.symphony.clients.model.SymMessage.)
    (.setMessage    m)
    (.setEntityData ed)))


(defn send-message!
  "Sends the given message (a String), optionally including entity data (a String containing JSON) to the given target (chat, room, or stream).

See:
  * https://rest-api.symphony.com/docs/messagemlv2 for details on MessageMLv2's formatting capabilities
  * https://rest-api.symphony.com/docs/objects for details on MessageMLv2's entity data capabilities"
  ([c t m] (send-message! c t m nil))
  ([^org.symphonyoss.client.SymphonyClient c t ^String m ^String ed]
    (.sendMessage (.getMessagesClient c)
                  (doto (org.symphonyoss.symphony.pod.model.Stream.)
                    (.setId (sys/stream-id t)))
                  (build-sym-message m ed))
    nil))


(defn register-listener
  "Registers f, a function with 1 parameter, as a message listener (callback), and returns a handle to that listener so that it can be deregistered later on, if needed.  Listeners registered in this manner are not scoped to any particular stream - they will be sent all messages from all streams that the authenticated connection user is a participant in.

The argument passed to f is a map generated by msgobj->map (see that fn for details).

The value returned by f (if any) is ignored."
  [^org.symphonyoss.client.SymphonyClient c f]
  (let [listener (reify
                   org.symphonyoss.client.services.MessageListener
                   (onMessage [this msg]
                     (f (msgobj->map msg))))]
    (.addMessageListener (.getMessageService c) listener)
    listener))


(defn deregister-listener
  "Deregisters a previously-registered message listener.  Once deregistered, a listener should be discarded (they cannot be reused). Returns true if a valid message listener was deregistered, false otherwise."
  [^org.symphonyoss.client.SymphonyClient c ^org.symphonyoss.client.services.MessageListener l]
  (.removeMessageListener (.getMessageService c) l))
