;
; Copyright © 2017 Symphony Software Foundation
; SPDX-License-Identifier: Apache-2.0
;
; Licensed under the Apache License, Version 2.0 (the "License");
; you may not use this file except in compliance with the License.
; You may obtain a copy of the License at
;
;     http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.
;

(ns clj-symphony.message
  "Operations related to messages.  Currently, Symphony supports these message formats:
  1. plain text
  2. messageML, which supports a small number of formatting tags
  3. MessageMLv2 (as of Symphony v1.46)"
  (:require [clojure.string      :as s]
            [clj-symphony.user   :as syu]
            [clj-symphony.stream :as sys]))


(defn msgobj->map
  "Converts a SymMessage object into a map."
  [^org.symphonyoss.symphony.clients.model.SymMessage message]
  (if message
    {
      :message-id     (.getId          message)
      :timestamp      (java.util.Date. (Long/valueOf (.getTimestamp message)))
      :stream-id      (.getStreamId    message)
      :user-id        (.getFromUserId  message)
; "Formats are being phased out by Symphony."
;      :message-format (when-not (nil? (.getFormat message))
;                        (keyword (str (.getFormat message))))
      :type           (.getMessageType message)   ; This seems to be null or blank most of the time...
      :text           (.getMessage     message)
      :attachment     (.getAttachment  message)
      :entity-data    (.getEntityData  message)
    }))


(comment "Formats are being phased out by Symphony."
(def message-formats
  "The set of possible message formats in Symphony, as keywords."
  (set (map #(keyword (str %)) (org.symphonyoss.symphony.clients.model.SymMessage$Format/values))))
)

(defn escape
  "Escapes the given string for MessageML."
  [^String s]
  (when s
    (org.apache.commons.lang3.StringEscapeUtils/escapeXml11 s)))


(defn- ^org.symphonyoss.symphony.clients.model.SymMessage build-sym-message
  [^String message ^String entity-data]
  (let [msg (doto
              (org.symphonyoss.symphony.clients.model.SymMessage.)
              (.setMessage    message)
              (.setEntityData entity-data))]
(comment "Formats are being phased out by Symphony."
    (if (.startsWith message "<messageML>")
      (.setFormat msg org.symphonyoss.symphony.clients.model.SymMessage$Format/MESSAGEML)
      (.setFormat msg org.symphonyoss.symphony.clients.model.SymMessage$Format/TEXT))
)
    msg))


(defn send-message!
  "Sends the given message (a String), optionally including entity data (a String containing JSON) to the given target (chat, room, or stream).

See:
  * https://rest-api.symphony.com/docs/messagemlv2 for details on MessageMLv2's formatting capabilities
  * https://rest-api.symphony.com/docs/objects for details on MessageMLv2's entity data capabilities"
  ([connection target message] (send-message! connection target message nil))
  ([^org.symphonyoss.client.SymphonyClient connection target ^String message ^String entity-data]
   (let [stream-id (sys/stream-id target)
         stream    (doto (org.symphonyoss.symphony.pod.model.Stream.)
                     (.setId stream-id))]
     (.sendMessage (.getMessagesClient connection)
                   stream
                   (build-sym-message message entity-data))
     nil)))


(defn register-listener
  "Registers f, a function with 1 parameter, as a message listener (callback), and returns a handle to that listener so that it can be deregistered later on, if needed.  Listeners registered in this manner are not scoped to any particular stream - they will be sent all messages from all streams that the authenticated connection user is a participant in.

The argument passed to f is a map generated by msgobj->map (see that fn for details).

The value returned by f (if any) is ignored."
  [^org.symphonyoss.client.SymphonyClient connection f]
  (let [listener (reify
                   org.symphonyoss.client.services.MessageListener
                   (onMessage [this msg]
                     (f (msgobj->map msg))))]
    (.addMessageListener (.getMessageService connection) listener)
    listener))


(defn deregister-listener
  "Deregisters a previously-registered message listener.  Once deregistered, a listener should be discarded (they cannot be reused). Returns true if a valid message listener was deregistered, false otherwise."
  [^org.symphonyoss.client.SymphonyClient connection ^org.symphonyoss.client.services.MessageListener listener]
  (.removeMessageListener (.getMessageService connection) listener))
