(ns org.soulspace.qclojure.application.algorithm.qaoa
  "Quantum Approximate Optimization Algorithm (QAOA) Implementation
  
  QAOA is a quantum-classical hybrid algorithm for solving combinatorial optimization
  problems. It uses alternating application of a problem Hamiltonian and a mixer
  Hamiltonian to explore the solution space and find approximate solutions.
  
  Key Features:
  - Problem Hamiltonian construction for MaxCut, Max-SAT, and TSP
  - Mixer Hamiltonian (typically X-gates on all qubits)  
  - Alternating evolution structure: U(β,γ) = e^(-iβH_M)e^(-iγH_P)
  - Integration with classical optimization for parameter tuning
  - Performance analysis and approximation ratio calculation
  
  Algorithm Flow:
  1. Initialize equal superposition state |+⟩^n
  2. Apply p layers of alternating evolution U(β_i,γ_i)
  3. Measure expectation value of the problem Hamiltonian
  4. Use classical optimizer to update parameters (β,γ)
  5. Repeat until convergence or maximum iterations
  
  The algorithm targets NP-hard combinatorial optimization problems and provides
  quantum advantage for certain problem instances."
  (:require [clojure.spec.alpha :as s]
            [fastmath.core :as m]
            [org.soulspace.qclojure.domain.circuit :as qc]
            [org.soulspace.qclojure.domain.circuit-composition :as cc]
            [org.soulspace.qclojure.domain.state :as qs]
            [org.soulspace.qclojure.domain.hamiltonian :as ham]
            [org.soulspace.qclojure.application.algorithm.optimization :as qopt]
            [org.soulspace.qclojure.application.backend :as qb]))

;;
;; Specs for QAOA components
;;
(s/def ::qaoa-problem 
  #{:max-cut :max-sat :tsp :custom})

(s/def ::graph 
  (s/coll-of (s/tuple pos-int? pos-int? number?)))  ; [(i j weight), ...]

(s/def ::qaoa-config
  (s/keys :req-un [::problem-hamiltonian ::num-qubits ::num-layers]
          :opt-un [::mixer-hamiltonian ::initial-parameters ::max-iterations 
                   ::tolerance ::optimization-method ::shots ::graph
                   ::problem-instance]))

(s/def ::num-layers pos-int?)
(s/def ::problem-instance any?)  ; Problem-specific data structure

;;
;; Problem Hamiltonian Construction
;;
(defn max-cut-hamiltonian
  "Create a MaxCut problem Hamiltonian for a given graph.
  
  The MaxCut problem aims to find a partition of vertices that maximizes
  the number of edges crossing the partition. The problem Hamiltonian is:
  
  H_P = Σ_{(i,j)∈E} w_{ij} * (1 - Z_i Z_j) / 2
  
  Where w_{ij} is the edge weight between vertices i and j.
  The ground state corresponds to the maximum cut.
  
  Parameters:
  - graph: Collection of edges as [vertex1 vertex2 weight] tuples
  - num-vertices: Number of vertices in the graph (determines number of qubits)
  
  Returns:
  Collection of Pauli terms representing the MaxCut Hamiltonian
  
  Example:
  (max-cut-hamiltonian [[0 1 1.0] [1 2 2.0] [0 2 1.5]] 3)
  ;=> Hamiltonian for triangle graph with weighted edges"
  [graph num-vertices]
  {:pre [(coll? graph) (pos-int? num-vertices)
         (every? #(and (= (count %) 3) 
                      (< (first %) num-vertices)
                      (< (second %) num-vertices)
                      (number? (nth % 2))) graph)]}
  (mapcat (fn [[i j weight]]
            ;; Each edge contributes: weight * (1 - Z_i Z_j) / 2
            ;; = weight/2 * I - weight/2 * Z_i Z_j
            (let [coeff (/ weight 2.0)
                  identity-string (apply str (repeat num-vertices \I))
                  zz-string (-> (vec (repeat num-vertices \I))
                               (assoc i \Z)
                               (assoc j \Z)
                               (->> (apply str)))]
              [{:coefficient coeff :pauli-string identity-string}
               {:coefficient (- coeff) :pauli-string zz-string}]))
          graph))

(defn max-sat-hamiltonian
  "Create a Max-SAT problem Hamiltonian for a given Boolean formula.
  
  The Max-SAT problem aims to find a Boolean assignment that satisfies
  the maximum number of clauses. The problem Hamiltonian is:
  
  H_P = Σ_c (1 - P_c) / 2
  
  Where P_c is the projector onto the subspace where clause c is satisfied.
  For a clause with variables x_i, x_j (positive) and ¬x_k (negative):
  P_c = (1 + Z_i)/2 ∨ (1 + Z_j)/2 ∨ (1 - Z_k)/2
  
  Parameters:
  - clauses: Collection of clauses, each clause is a vector of literals
             Positive literal: variable index, Negative literal: negative index
  - num-variables: Number of Boolean variables (determines number of qubits)
  
  Returns:
  Collection of Pauli terms representing the Max-SAT Hamiltonian
  
  Example:
  (max-sat-hamiltonian [[0 1] [-0 2] [1 -2]] 3)
  ;=> Hamiltonian for (x₀ ∨ x₁) ∧ (¬x₀ ∨ x₂) ∧ (x₁ ∨ ¬x₂)"
  [clauses num-variables]
  {:pre [(coll? clauses) (pos-int? num-variables)
         (every? #(and (coll? %) (seq %)) clauses)]}
  (mapcat (fn [clause]
            ;; For each clause, we need to construct the projector
            ;; This is a simplified version - full implementation requires
            ;; more complex decomposition of the OR operation
            (let [num-literals (count clause)]
              ;; Simplified: each unsatisfied clause contributes 1/2
              ;; In practice, this needs proper decomposition of the OR constraint
              [{:coefficient 0.5 :pauli-string (apply str (repeat num-variables \I))}]))
          clauses))

(defn travelling-salesman-hamiltonian
  "Create a Travelling Salesman Problem (TSP) Hamiltonian.
  
  The TSP aims to find the shortest route visiting all cities exactly once.
  This is encoded using n² qubits where qubit (i,j) represents whether
  city i is visited at time j.
  
  The Hamiltonian includes:
  1. Cost function: minimize total distance
  2. Constraints: each city visited exactly once
  3. Constraints: exactly one city visited at each time
  
  Parameters:
  - distance-matrix: n×n matrix of distances between cities
  
  Returns:
  Collection of Pauli terms representing the TSP Hamiltonian
  
  Note: This is a complex encoding requiring detailed constraint decomposition"
  [distance-matrix]
  {:pre [(coll? distance-matrix)
         (let [n (count distance-matrix)]
           (and (pos? n)
                (every? #(= (count %) n) distance-matrix)))]}
  (let [n (count distance-matrix)
        num-qubits (* n n)]
    ;; Simplified placeholder - full TSP Hamiltonian is quite complex
    ;; Would need proper constraint decomposition
    [{:coefficient 1.0 :pauli-string (apply str (repeat num-qubits \I))}]))

(defn custom-ising-hamiltonian
  "Create a custom Ising model Hamiltonian.
  
  General Ising model: H = Σᵢ hᵢZᵢ + Σᵢⱼ Jᵢⱼ ZᵢZⱼ
  
  Parameters:
  - h-fields: Vector of local magnetic fields [h₀ h₁ ... hₙ₋₁]
  - j-couplings: Collection of coupling terms [[i j Jᵢⱼ], ...]
  
  Returns:
  Collection of Pauli terms representing the Ising Hamiltonian"
  [h-fields j-couplings]
  {:pre [(vector? h-fields) (coll? j-couplings)]}
  (let [n (count h-fields)]
    (concat
     ;; Local field terms: hᵢZᵢ
     (map-indexed (fn [i h]
                    (let [z-string (-> (vec (repeat n \I))
                                      (assoc i \Z)
                                      (->> (apply str)))]
                      {:coefficient h :pauli-string z-string}))
                  h-fields)
     ;; Coupling terms: JᵢⱼZᵢZⱼ
     (map (fn [[i j coupling]]
            (let [zz-string (-> (vec (repeat n \I))
                               (assoc i \Z)
                               (assoc j \Z)
                               (->> (apply str)))]
              {:coefficient coupling :pauli-string zz-string}))
          j-couplings))))

;;
;; Mixer Hamiltonian Construction
;;
(defn standard-mixer-hamiltonian
  "Create the standard QAOA mixer Hamiltonian.
  
  The mixer Hamiltonian is typically H_M = Σᵢ Xᵢ, which creates
  transitions between computational basis states and allows exploration
  of the solution space.
  
  Parameters:
  - num-qubits: Number of qubits in the system
  
  Returns:
  Collection of Pauli terms representing the mixer Hamiltonian"
  [num-qubits]
  {:pre [(pos-int? num-qubits)]}
  (map (fn [i]
         (let [x-string (-> (vec (repeat num-qubits \I))
                           (assoc i \X)
                           (->> (apply str)))]
           {:coefficient 1.0 :pauli-string x-string}))
       (range num-qubits)))

(defn xy-mixer-hamiltonian
  "Create an XY mixer Hamiltonian for problems with particle number conservation.
  
  The XY mixer preserves the number of |1⟩ states and is useful for problems
  where the constraint is to select exactly k items out of n.
  
  H_M = Σᵢ (Xᵢ₊₁Xᵢ + Yᵢ₊₁Yᵢ) = Σᵢ (σᵢ₊ σᵢ₋ + σᵢ₋ σᵢ₊₁)
  
  Parameters:
  - num-qubits: Number of qubits in the system
  - periodic: Whether to include periodic boundary conditions
  
  Returns:
  Collection of Pauli terms representing the XY mixer Hamiltonian"
  [num-qubits periodic]
  {:pre [(pos-int? num-qubits)]}
  (let [pairs (if periodic
                (map vector (range num-qubits) (map #(mod % num-qubits) (range 1 (inc num-qubits))))
                (map vector (range (dec num-qubits)) (range 1 num-qubits)))]
    (mapcat (fn [[i j]]
              (let [xx-string (-> (vec (repeat num-qubits \I))
                                 (assoc i \X)
                                 (assoc j \X)
                                 (->> (apply str)))
                    yy-string (-> (vec (repeat num-qubits \I))
                                 (assoc i \Y)
                                 (assoc j \Y)
                                 (->> (apply str)))]
                [{:coefficient 1.0 :pauli-string xx-string}
                 {:coefficient 1.0 :pauli-string yy-string}]))
            pairs)))

;;
;; QAOA Circuit Construction
;;
(defn hamiltonian-evolution-circuit
  "Create a quantum circuit for time evolution under a Hamiltonian.
  
  For a Hamiltonian H and evolution time t, this creates the circuit
  implementing e^(-iHt). For Pauli string Hamiltonians, this decomposes
  into products of rotations and CNOT gates.
  
  Parameters:
  - hamiltonian: Collection of Pauli terms
  - evolution-time: Evolution time parameter
  - num-qubits: Number of qubits
  
  Returns:
  Quantum circuit implementing the evolution"
  [hamiltonian evolution-time num-qubits]
  {:pre [(ham/validate-hamiltonian hamiltonian) (number? evolution-time) (pos-int? num-qubits)]}
  (let [circuit (qc/create-circuit num-qubits "Hamiltonian Evolution")]
    (reduce (fn [circ term]
              (let [coeff (:coefficient term)
                    pauli-str (:pauli-string term)
                    angle (* 2.0 evolution-time coeff)]  ; Factor of 2 for rotation angle
                (cond
                  ;; Identity terms contribute only global phase (can be ignored)
                  (every? #(= % \I) pauli-str)
                  circ
                  
                  ;; Single Pauli terms: direct rotation
                  (= 1 (count (remove #(= % \I) pauli-str)))
                  (let [qubit-idx (first (keep-indexed #(when (not= %2 \I) %1) pauli-str))
                        pauli-op (nth pauli-str qubit-idx)]
                    (case pauli-op
                      \X (qc/rx-gate circ qubit-idx angle)
                      \Y (qc/ry-gate circ qubit-idx angle)
                      \Z (qc/rz-gate circ qubit-idx angle)
                      circ))  ; Default case for unknown Pauli operators
                  
                  ;; Multi-Pauli terms: use CNOT ladder for ZZ interactions
                  :else
                  (let [pauli-positions (keep-indexed #(when (not= %2 \I) [%1 %2]) pauli-str)]
                    ;; For ZZ terms, we can use direct RZ with CNOT ladders
                    ;; For more complex terms, we'd need more sophisticated decomposition
                    (if (every? #(= (second %) \Z) pauli-positions)
                      ;; ZZ interaction: create CNOT ladder, apply RZ to last qubit, uncompute
                      (let [qubit-indices (mapv first pauli-positions)]
                        (if (= (count qubit-indices) 2)
                          ;; Two-qubit ZZ: simple case
                          (let [[q1 q2] qubit-indices]
                            (-> circ
                                (qc/cnot-gate q1 q2)
                                (qc/rz-gate q2 angle)
                                (qc/cnot-gate q1 q2)))
                          ;; Multi-qubit ZZ: use CNOT ladder
                          (let [target (last qubit-indices)
                                controls (butlast qubit-indices)
                                c-with-cnots 
                                (reduce (fn [c ctrl]
                                          (qc/cnot-gate c ctrl target))
                                        circ
                                        controls)]
                            (-> c-with-cnots
                                (qc/rz-gate target angle)
                                ;; Uncompute CNOT ladder
                                (#(reduce (fn [c ctrl]
                                            (qc/cnot-gate c ctrl target))
                                          %
                                          (reverse controls)))))))
                      ;; For X/Y terms, would need more complex decomposition
                      circ)))))
            circuit
            hamiltonian)))

(defn qaoa-ansatz-circuit
  "Create a QAOA ansatz circuit with alternating problem and mixer evolution.
  
  The QAOA ansatz consists of p layers of alternating evolution:
  U(β,γ) = ∏ᵢ₌₁ᵖ e^(-iβᵢH_M) e^(-iγᵢH_P)
  
  Where H_P is the problem Hamiltonian and H_M is the mixer Hamiltonian.
  
  Parameters:
  - problem-hamiltonian: Problem Hamiltonian (collection of Pauli terms)
  - mixer-hamiltonian: Mixer Hamiltonian (collection of Pauli terms)
  - parameters: Vector of [γ₁ β₁ γ₂ β₂ ... γₚ βₚ] parameters
  - num-qubits: Number of qubits
  
  Returns:
  Quantum circuit implementing the QAOA ansatz"
  [problem-hamiltonian mixer-hamiltonian parameters num-qubits]
  {:pre [(ham/validate-hamiltonian problem-hamiltonian)
         (ham/validate-hamiltonian mixer-hamiltonian)
         (vector? parameters)
         (even? (count parameters))
         (pos-int? num-qubits)]}
  (let [p (/ (count parameters) 2)  ; Number of QAOA layers
        param-pairs (partition 2 parameters)  ; [(γ₁ β₁) (γ₂ β₂) ...]
        initial-circuit (qc/create-circuit num-qubits "QAOA Ansatz")]
    
    ;; Start with equal superposition state |+⟩^⊗n
    (let [circuit-with-init (reduce (fn [circ qubit]
                                      (qc/h-gate circ qubit))
                                    initial-circuit
                                    (range num-qubits))]
      
      ;; Apply p layers of alternating evolution
      (reduce (fn [circ [gamma beta]]
                (-> circ
                    ;; Apply problem Hamiltonian evolution e^(-iγH_P)
                    (#(let [prob-circuit (hamiltonian-evolution-circuit problem-hamiltonian gamma num-qubits)]
                        (cc/compose-circuits % prob-circuit)))
                    ;; Apply mixer Hamiltonian evolution e^(-iβH_M)  
                    (#(let [mixer-circuit (hamiltonian-evolution-circuit mixer-hamiltonian beta num-qubits)]
                        (cc/compose-circuits % mixer-circuit)))))
              circuit-with-init
              param-pairs))))

;;
;; QAOA Algorithm Implementation
;;
(defn create-qaoa-objective
  "Create the objective function for QAOA optimization.
  
  This function creates the objective function that:
  1. Takes QAOA parameters [γ₁ β₁ γ₂ β₂ ...] as input
  2. Constructs the QAOA ansatz circuit with those parameters
  3. Executes the circuit on the backend or simulator
  4. Calculates the problem Hamiltonian expectation value
  5. Returns the energy (to be minimized by the optimizer)
  
  Parameters:
  - problem-hamiltonian: Problem Hamiltonian to minimize
  - mixer-hamiltonian: Mixer Hamiltonian (optional, defaults to standard X mixer)
  - num-qubits: Number of qubits
  - backend: Quantum backend for circuit execution (can be nil for direct simulation)
  - options: Execution options (shots, etc.)
  
  Returns:
  Function that takes parameters and returns energy expectation value"
  [problem-hamiltonian mixer-hamiltonian num-qubits backend options]
  {:pre [(ham/validate-hamiltonian problem-hamiltonian)
         (ham/validate-hamiltonian mixer-hamiltonian)
         (pos-int? num-qubits)]}
  (fn objective [parameters]
    (try
      (let [circuit (qaoa-ansatz-circuit problem-hamiltonian mixer-hamiltonian parameters num-qubits)]
        (if backend
          ;; Execute on quantum backend
          (let [result (qb/execute-circuit backend circuit options)
                final-state (:final-state result)]
            (ham/hamiltonian-expectation problem-hamiltonian final-state))
          ;; Direct simulation
          (let [initial-state (qs/zero-state num-qubits)
                final-state (qc/execute-circuit circuit initial-state)]
            (ham/hamiltonian-expectation problem-hamiltonian final-state))))
      (catch Exception e
        (println "Error in QAOA objective function:" (.getMessage e))
        Double/POSITIVE_INFINITY))))  ; Return high energy for failed evaluations

(defn qaoa-optimization
  "Run QAOA optimization using the specified method.
  
  Similar to VQE optimization but specialized for QAOA's alternating structure.
  Supports the same optimization methods as VQE.
  
  Parameters:
  - objective-fn: QAOA objective function to minimize
  - initial-parameters: Starting parameter values [γ₁ β₁ γ₂ β₂ ...]
  - options: Optimization options
  
  Returns:
  Map with optimization results"
  [objective-fn initial-parameters options]
  (let [method (:optimization-method options :adam)]  ; Default to Adam
    (case method
      ;; Custom implementations with parameter shift rules
      :gradient-descent 
      (qopt/gradient-descent-optimization objective-fn initial-parameters options)
      
      :adam
      (qopt/adam-optimization objective-fn initial-parameters options)
      
      :quantum-natural-gradient
      (qopt/quantum-natural-gradient-optimization objective-fn initial-parameters 
                                            (merge options
                                                   {:ansatz-fn (:ansatz-fn options)
                                                    :backend (:backend options)
                                                    :exec-options (:exec-options options)}))
      
      ;; Fastmath derivative-free optimizers (verified working)
      (:nelder-mead :powell :cmaes :bobyqa)
      (qopt/fastmath-derivative-free-optimization method objective-fn initial-parameters options)
      
      ;; Fastmath gradient-based optimizers
      (:gradient :lbfgsb)
      (qopt/fastmath-gradient-based-optimization method objective-fn initial-parameters options)

      ;; Default fallback
      (throw (ex-info "Unknown optimization method" {:method method})))))

(defn quantum-approximate-optimization-algorithm
  "Main QAOA algorithm implementation.

  This function orchestrates the QAOA process, including circuit construction,
  optimization, and execution on a quantum backend.
  
  Supported problem types:
  - :max-cut - Maximum cut problem on graphs
  - :max-sat - Maximum satisfiability problem
  - :tsp - Travelling salesman problem (simplified encoding)
  - :custom - Custom problem with provided Hamiltonian
   
  Supported optimization methods:
  - :gradient-descent - Basic gradient descent with parameter shift gradients
  - :adam - Adam optimizer with parameter shift gradients (recommended default)
  - :quantum-natural-gradient - Quantum Natural Gradient using Fisher Information Matrix
  - :nelder-mead - Derivative-free Nelder-Mead simplex method
  - :powell - Derivative-free Powell's method
  - :cmaes - Covariance Matrix Adaptation Evolution Strategy (robust)
  - :bobyqa - Bound Optimization BY Quadratic Approximation (handles bounds well)
  
  Parameters:
  - backend: Quantum backend implementing QuantumBackend protocol
  - options: QAOA configuration options
    - :problem-type - Type of problem (:max-cut, :max-sat, :tsp, :custom)
    - :problem-instance - Problem-specific data (graph, clauses, distance matrix, etc.)
    - :num-qubits - Number of qubits in the system
    - :num-layers - Number of QAOA layers (p parameter)
    - :optimization-method - Optimization method to use (default: :adam)
    - :max-iterations - Maximum iterations for optimization (default: 500)
    - :tolerance - Convergence tolerance (default: 1e-6)
    - :shots - Number of shots for circuit execution (default: 1024)
    - :initial-parameters - Custom initial parameters (optional)
  
  Returns:
  Map containing QAOA results and analysis
  
  Example:
  (quantum-approximate-optimization-algorithm backend
    {:problem-type :max-cut
     :problem-instance [[0 1 1.0] [1 2 1.0] [0 2 1.0]]  ; Triangle graph
     :num-qubits 3
     :num-layers 2
     :optimization-method :adam
     :max-iterations 100})"
  [backend options]
  {:pre [(s/valid? ::qaoa-config options)]}
  (let [problem-type (:problem-type options)
        problem-instance (:problem-instance options)
        num-qubits (:num-qubits options)
        num-layers (:num-layers options)
        
        ;; Create problem Hamiltonian based on type
        problem-hamiltonian 
        (case problem-type
          :max-cut (max-cut-hamiltonian problem-instance num-qubits)
          :max-sat (max-sat-hamiltonian problem-instance num-qubits)
          :tsp (travelling-salesman-hamiltonian problem-instance)
          :custom (:problem-hamiltonian options)
          (throw (ex-info "Unknown problem type" {:problem-type problem-type})))
        
        ;; Create mixer Hamiltonian (default: standard X mixer)
        mixer-hamiltonian (or (:mixer-hamiltonian options)
                             (standard-mixer-hamiltonian num-qubits))
        
        ;; Initial parameters: small random values for [γ₁ β₁ γ₂ β₂ ...]
        initial-parameters (or (:initial-parameters options)
                              (vec (repeatedly (* 2 num-layers) 
                                               #(* 0.1 (- (rand) 0.5)))))
        
        ;; Create objective function
        exec-options {:shots (:shots options 1024)}
        objective-fn (create-qaoa-objective problem-hamiltonian mixer-hamiltonian 
                                           num-qubits backend exec-options)
        
        ;; Run optimization
        optimization-options (merge options {:gradient-method :parameter-shift})
        start-time (System/currentTimeMillis)
        optimization-result (qaoa-optimization objective-fn initial-parameters optimization-options)
        end-time (System/currentTimeMillis)
        
        ;; Build final circuit with optimal parameters
        optimal-params (:optimal-parameters optimization-result)
        final-circuit (qaoa-ansatz-circuit problem-hamiltonian mixer-hamiltonian 
                                          optimal-params num-qubits)
        
        ;; Calculate final energy and additional metrics
        final-energy (:optimal-energy optimization-result)
        approximation-ratio (when (:classical-optimum options)
                             (/ final-energy (:classical-optimum options)))]
    
    (merge optimization-result
           {:algorithm "QAOA"
            :problem-type problem-type
            :num-qubits num-qubits
            :num-layers num-layers
            :problem-hamiltonian problem-hamiltonian
            :mixer-hamiltonian mixer-hamiltonian
            :final-circuit final-circuit
            :total-runtime-ms (- end-time start-time)
            :approximation-ratio approximation-ratio
            :gamma-parameters (vec (take-nth 2 optimal-params))      ; [γ₁ γ₂ ...]
            :beta-parameters (vec (take-nth 2 (rest optimal-params))) ; [β₁ β₂ ...]
            })))

;;
;; Analysis and Utilities
;;
(defn analyze-qaoa-parameters
  "Analyze QAOA parameter landscape and patterns.
  
  Parameters:
  - optimization-result: Result from QAOA optimization
  
  Returns:
  Map with parameter analysis"
  [optimization-result]
  (let [optimal-params (:optimal-parameters optimization-result)
        gamma-params (:gamma-parameters optimization-result)
        beta-params (:beta-parameters optimization-result)]
    
    {:parameter-summary
     {:gamma-range [(apply min gamma-params) (apply max gamma-params)]
      :beta-range [(apply min beta-params) (apply max beta-params)]
      :gamma-mean (/ (reduce + gamma-params) (count gamma-params))
      :beta-mean (/ (reduce + beta-params) (count beta-params))
      :total-parameters (count optimal-params)}
     
     :convergence-analysis
     (when-let [history (:convergence-history optimization-result)]
       (let [energies (vec history)
             improvement (when (> (count energies) 1)
                          (- (first energies) (last energies)))]
         {:initial-energy (first energies)
          :final-energy (last energies)
          :total-improvement improvement
          :convergence-achieved (:success optimization-result)}))
     
     :performance-metrics
     {:function-evaluations (:function-evaluations optimization-result 0)
      :total-runtime-ms (:total-runtime-ms optimization-result)
      :approximation-ratio (:approximation-ratio optimization-result)}}))

(defn estimate-classical-optimum
  "Estimate the classical optimum for comparison with QAOA results.
  
  For small problem instances, this performs brute-force enumeration.
  For larger instances, this provides heuristic estimates.
  
  Parameters:
  - problem-type: Type of optimization problem
  - problem-instance: Problem-specific data
  - num-qubits: Number of qubits (determines search space size)
  
  Returns:
  Estimated classical optimum value"
  [problem-type problem-instance num-qubits]
  (case problem-type
    :max-cut
    (if (<= num-qubits 12)  ; Brute force for small instances
      (let [graph problem-instance]
        ;; Evaluate all 2^n possible cuts
        (reduce max
                (for [assignment (range (bit-shift-left 1 num-qubits))]
                  (reduce +
                          (for [[i j weight] graph]
                            (let [bit-i (bit-test assignment i)
                                  bit-j (bit-test assignment j)]
                              (if (not= bit-i bit-j) weight 0.0)))))))
      ;; Heuristic for larger instances
      (let [total-weight (reduce + (map #(nth % 2) problem-instance))]
        (* 0.5 total-weight)))  ; Rough estimate: half of total edge weight
    
    :max-sat
    ;; For Max-SAT, would need proper SAT solver integration
    (count problem-instance)  ; Optimistic: all clauses satisfied
    
    :tsp
    ;; For TSP, would need proper heuristic or exact solver
    1.0  ; Placeholder
    
    :custom
    nil))  ; Cannot estimate without knowing the problem structure

(comment
  (require '[org.soulspace.qclojure.adapter.backend.simulator :as sim])
  
  ;; Example 1: MaxCut on triangle graph
  (def triangle-maxcut-config
    {:problem-type :max-cut
     :problem-instance [[0 1 1.0] [1 2 1.0] [0 2 1.0]]  ; Triangle with unit weights
     :num-qubits 3
     :num-layers 2
     :optimization-method :adam
     :max-iterations 100
     :tolerance 1e-6})
  
  (def backend (sim/create-simulator))
  (def qaoa-result (quantum-approximate-optimization-algorithm backend triangle-maxcut-config))
  
  ;; Analyze results
  (def analysis (analyze-qaoa-parameters qaoa-result))
  (def classical-opt (estimate-classical-optimum :max-cut (:problem-instance triangle-maxcut-config) 3))
  
  ;; Print key results
  (println "QAOA Energy:" (:optimal-energy qaoa-result))
  (println "Classical Optimum:" classical-opt)
  (println "Approximation Ratio:" (/ (:optimal-energy qaoa-result) classical-opt))
  (println "Optimal γ parameters:" (:gamma-parameters qaoa-result))
  (println "Optimal β parameters:" (:beta-parameters qaoa-result))
  
  ;; Example 2: Custom Ising model
  (def ising-config
    {:problem-type :custom
     :problem-hamiltonian (custom-ising-hamiltonian [0.5 -0.3 0.2] [[0 1 0.1] [1 2 -0.2]])
     :num-qubits 3
     :num-layers 1
     :optimization-method :nelder-mead
     :max-iterations 50})
  
  )
