;;;;
;;;;   Copyright (c) Ludger Solbach. All rights reserved.
;;;;
;;;;   The use and distribution terms for this software are covered by the
;;;;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;;;;   which can be found in the file license.txt at the root of this distribution.
;;;;   By using this software in any fashion, you are agreeing to be bound by
;;;;   the terms of this license.
;;;;
;;;;   You must not remove this notice, or any other, from this software.
;;;;

(ns org.soulspace.tools.project-info
  (:require [clojure.java.io :as io]
            [clojure.data.json :as json]
            [clojure.data.xml :as xml]))

;;;;
;;;; external project information
;;;;

;;;
;;; Github handling
;;;

(def github-pattern #"https://github.com/(.*)")
(def github-root "https://github.com")

(defn github-url?
  "Returns truthy, if the url is a github project url."
  [url]
  (re-matches github-pattern url))

(defn github-releases-url
  "Returns the URL for the releases of the project."
  [p]
  (str github-root "/" p "/releases"))

(defn github-contributors-url
  "Returns the URL for contributors graph of the project."
  [p]
  (str github-root "/" p "/graphs/contributors"))

(comment
  (github-releases-url "lsolbach/CljBase")
  (github-contributors-url "lsolbach/CljBase"))

;;;
;;; Apache handling
;;;

(def apache-pattern #"https://(.*)\.apache\.org.*")

(defn apache-url?
  "Returns truthy, if the url is a apache project url."
  [url]
  (re-matches apache-pattern url))

;;;
;;; OpenHub handling
;;;

;; Docs
; https://github.com/blackducksoftware/ohloh_api
; https://github.com/blackducksoftware/ohloh_api/blob/master/reference/project.md

;; URLs
; https://www.openhub.net/projects.xml
; https://www.openhub.net/projects/{project_id}.xml

(def openhub-root "https://www.openhub.net")
(def openhub-api-key "") ; TODO set from ENV

(defn openhub-projects-xml
  "Fetches the projects from openhub.net."
  []
  (slurp (str openhub-root "/projects.xml"))
;  (-> (slurp (str openhub-root "/projects.xml"))
;      (xml/parse))
  )

(defn openhub-project-xml
  "Fetches the project from openhub.net."
  [id]
  (with-open [rdr (io/reader (slurp (str openhub-root "/projects/" id ".xml")))]
      (doall (xml/parse rdr))))

(comment
 (defn openhub-projects-xml
  "Fetches the projects from openhub.net."
  []
  (with-open [input (io/input-stream
                     (io/as-url (str openhub-root "/projects.xml")))]
      (doall (xml/parse input))))

(defn openhub-project-xml
  "Fetches the project from openhub.net."
  [id]
  (with-open [input (io/input-stream
                     (io/as-url (str openhub-root "/projects/" id ".xml")))]
    (doall (xml/parse input))))
)

(comment
  (openhub-projects-xml))