(ns org.soulspace.arrayfire.integration.moments
  "Integration of the ArrayFire moments related FFI bindings with the error
   handling and resource management on the JVM.
   
   This namespace provides image moment calculation for computer vision and
   image analysis tasks. Moments are quantitative measures of shape and
   intensity distribution, fundamental to object localization and tracking.
   
   ## Image Moments Overview
   
   An image moment is a weighted average of pixel intensities:
   ```
   M_pq = Σ_x Σ_y x^p * y^q * I(x,y)
   ```
   
   ## First-Order Moments
   
   - **M00**: Zeroth moment (total mass/area)
     * Sum of all pixel intensities
     * For binary images: number of foreground pixels
     
   - **M01**: First moment about y-axis
     * X-weighted sum → centroid x-coordinate: x̄ = M01/M00
     
   - **M10**: First moment about x-axis
     * Y-weighted sum → centroid y-coordinate: ȳ = M10/M00
     
   - **M11**: Second mixed moment
     * Used for orientation and shape analysis
   
   ## Common Applications
   
   1. **Object Localization**: Find centroid (center of mass)
   2. **Shape Analysis**: Compute area, orientation, elongation
   3. **Tracking**: Track object center across frames
   4. **Recognition**: Moment-based features for classification
   5. **Registration**: Align images by matching centroids
   
   ## Centroid Calculation
   
   The most common use of moments is finding the centroid:
   ```clojure
   (let [{:keys [M00 M01 M10]} (moments-first-order img)]
     (when (> M00 0)
       {:x (/ M01 M00)
        :y (/ M10 M00)
        :area M00}))
   ```"
  (:require [coffi.mem :as mem]
            [org.soulspace.arrayfire.ffi.moments :as moments-ffi]
            [org.soulspace.arrayfire.integration.jvm-integration :as jvm])
  (:import (org.soulspace.arrayfire.integration.jvm_integration AFArray)))

;;;
;;; Moment Type Constants
;;;

(def ^:const moment-m00 1)        ; Zeroth moment (total mass/area)
(def ^:const moment-m01 2)        ; First moment about y-axis
(def ^:const moment-m10 4)        ; First moment about x-axis
(def ^:const moment-m11 8)        ; Second mixed moment
(def ^:const moment-first-order 15) ; All four moments (1|2|4|8)

(defn moment-type->int
  "Convert moment type keyword to integer constant.
   
   Parameters:
   - type: Moment type (:m00, :m01, :m10, :m11, :first-order, or integer)
   
   Returns:
   Integer constant for moment type
   
   Examples:
   ```clojure
   (moment-type->int :m00) ; => 1
   (moment-type->int :first-order) ; => 15
   (moment-type->int #{:m00 :m01}) ; => 3
   ```"
  [type]
  (cond
    (keyword? type)
    (case type
      :m00 moment-m00
      :m01 moment-m01
      :m10 moment-m10
      :m11 moment-m11
      :first-order moment-first-order
      (throw (ex-info "Unknown moment type" {:type type})))
    
    (set? type)
    (reduce bit-or 0 (map moment-type->int type))
    
    (integer? type)
    type
    
    :else
    (throw (ex-info "Invalid moment type" {:type type}))))

;;;
;;; Image Moments
;;;

(defn moments
  "Calculate image moments for one or more images.
   
   Computes first-order moments (M00, M01, M10, M11) which provide
   quantitative measures of shape, position, and intensity distribution.
   Essential for object localization, tracking, and shape analysis.
   
   Parameters:
   - in: Input image(s), AFArray [width × height × batch_z × batch_w]
     * Supports batches: each image processed independently
     * Types: f32, f64, s32, u32, s16, u16, b8
   
   - moment-type: Type(s) of moments to compute
     * :m00 - Zeroth moment (total mass/area)
     * :m01 - First moment about y-axis (for x-centroid)
     * :m10 - First moment about x-axis (for y-centroid)
     * :m11 - Second mixed moment (for orientation)
     * :first-order - All four moments (most common)
     * #{:m00 :m01} - Multiple specific moments
     * Integer flag value (bitwise OR of moment constants)
   
   Returns:
   AFArray [num_moments × 1 × batch_z × batch_w] containing moments
   - num_moments = number of moment types requested (1-4)
   - Order: M00, M01, M10, M11 (in bit order)
   - Type: float (f32)
   
   Moment Definitions:
   
   **M00 - Total Mass/Area:**
   ```
   M00 = Σ_x Σ_y I(x,y)
   ```
   - Sum of all pixel intensities
   - For binary images: pixel count
   - Always ≥ 0
   
   **M01 - X-weighted Sum:**
   ```
   M01 = Σ_x Σ_y x * I(x,y)
   ```
   - Centroid x-coordinate: x̄ = M01 / M00
   
   **M10 - Y-weighted Sum:**
   ```
   M10 = Σ_x Σ_y y * I(x,y)
   ```
   - Centroid y-coordinate: ȳ = M10 / M00
   
   **M11 - XY-correlation:**
   ```
   M11 = Σ_x Σ_y x * y * I(x,y)
   ```
   - Used for orientation and shape analysis
   
   Examples:
   ```clojure
   ;; Compute all moments for single image
   (let [result (moments img :first-order)]
     ;; result[0] = M00, result[1] = M01, result[2] = M10, result[3] = M11
     result)
   
   ;; Compute only area (M00)
   (let [area-array (moments binary-img :m00)]
     ;; Extract M00 value
     (get-scalar area-array))
   
   ;; Compute moments needed for centroid
   (let [result (moments img #{:m00 :m01 :m10})]
     ;; result[0] = M00, result[1] = M01, result[2] = M10
     result)
   
   ;; Batch processing: compute centroids for 100 images
   (let [batch (create-batch-array images [512 512 1 100])
         moments (moments batch :first-order)]
     ;; moments shape: [4 × 1 × 1 × 100]
     ;; Extract centroids for each image...
     )
   ```
   
   Common Patterns:
   
   **Centroid Calculation:**
   ```clojure
   (defn centroid [img]
     (let [m (moments img :first-order)
           M00 (get-scalar m [0])
           M01 (get-scalar m [1])
           M10 (get-scalar m [2])]
       (when (> M00 0)
         [(/ M01 M00) (/ M10 M00)])))
   ```
   
   **Area Measurement:**
   ```clojure
   (defn area [binary-img]
     (let [m (moments binary-img :m00)]
       (get-scalar m)))
   ```
   
   **Batch Centroids:**
   ```clojure
   (defn batch-centroids [imgs]
     (let [m (moments imgs :first-order)]
       ;; Extract M00, M01, M10 for each image
       (for [i (range batch-size)]
         (let [M00 (get-scalar m [0 0 0 i])
               M01 (get-scalar m [1 0 0 i])
               M10 (get-scalar m [2 0 0 i])]
           [(/ M01 M00) (/ M10 M00)]))))
   ```
   
   Performance:
   - GPU accelerated: ~0.1 ms for 512×512 image
   - Batch processing: linear scaling
   - All moments computed in single pass
   
   See Also:
   - moments-all: Convenience function for single image returning map
   - centroid: Direct centroid calculation
   - area: Direct area calculation"
  [^AFArray in moment-type]
  (let [moment-int (moment-type->int moment-type)
        out (jvm/native-af-array-pointer)]
    (jvm/check! (moments-ffi/af-moments out (jvm/af-handle in) (int moment-int))
                "af-moments")
    (jvm/af-array-new (jvm/deref-af-array out))))

(defn moments-all
  "Calculate image moments for a single 2D image (convenience).
   
   Convenience function that computes moments and directly returns them
   as a map, avoiding the need to extract from an AFArray. Only works
   for single 2D images (not batches).
   
   Parameters:
   - in: Input 2D image, AFArray [width × height]
     * Must be 2D: dimensions [2] and [3] must equal 1
     * Types: f32, f64, s32, u32, s16, u16, b8
   
   - moment-type: Type(s) of moments (default :first-order)
     * Same options as `moments` function
   
   Returns:
   Map containing requested moments with double precision:
   - :M00 - Zeroth moment (if requested)
   - :M01 - First moment about y-axis (if requested)
   - :M10 - First moment about x-axis (if requested)
   - :M11 - Second mixed moment (if requested)
   
   Note: This function returns double precision values (f64) while
   the `moments` function returns float arrays (f32).
   
   Examples:
   ```clojure
   ;; Get all moments as map
   (let [{:keys [M00 M01 M10 M11]} (moments-all img)]
     (println \"Area:\" M00)
     (println \"Centroid:\" [(/ M01 M00) (/ M10 M00)]))
   
   ;; Get only area
   (let [{:keys [M00]} (moments-all binary-img :m00)]
     (println \"Pixels:\" M00))
   
   ;; Get centroid moments
   (let [{:keys [M00 M01 M10]} (moments-all img #{:m00 :m01 :m10})]
     {:x (/ M01 M00)
      :y (/ M10 M00)
      :area M00})
   ```
   
   Simple Centroid:
   ```clojure
   (defn compute-centroid [img]
     (let [{:keys [M00 M01 M10]} (moments-all img)]
       (when (> M00 0)
         {:x (/ M01 M00)
          :y (/ M10 M00)
          :area M00})))
   ```
   
   Object Detection:
   ```clojure
   (let [{:keys [M00 M01 M10]} (moments-all binary-mask)]
     (if (> M00 100)  ; At least 100 pixels
       {:detected true
        :center [(/ M01 M00) (/ M10 M00)]
        :pixels M00}
       {:detected false}))
   ```
   
   When to Use:
   - Single 2D image processing
   - Want simple map output
   - Don't need GPU array result
   - Convenience over performance
   
   Use `moments` instead when:
   - Processing batches
   - Chaining GPU operations
   - Performance critical
   
   Performance:
   - Slightly slower than `moments` due to host transfer
   - ~0.11 ms for 512×512 image (vs 0.10 ms)
   - Negligible difference for single images"
  ([^AFArray in]
   (moments-all in :first-order))
  ([^AFArray in moment-type]
   (let [moment-int (moment-type->int moment-type)
         ;; Allocate buffer for up to 4 moments (doubles)
         num-moments (Integer/bitCount moment-int)
         out-buf (mem/alloc (* num-moments 8))]
     (jvm/check! (moments-ffi/af-moments-all out-buf (jvm/af-handle in) (int moment-int))
                 "af-moments-all")
     ;; Read results and build map
     (let [result (transient {})]
       (when (bit-test moment-int 0) ; M00
         (assoc! result :M00 (mem/read-double out-buf 0)))
       (when (bit-test moment-int 1) ; M01
         (assoc! result :M01 (mem/read-double out-buf 
                                               (* 8 (Integer/bitCount (bit-and moment-int 1))))))
       (when (bit-test moment-int 2) ; M10
         (assoc! result :M10 (mem/read-double out-buf 
                                               (* 8 (Integer/bitCount (bit-and moment-int 3))))))
       (when (bit-test moment-int 3) ; M11
         (assoc! result :M11 (mem/read-double out-buf 
                                               (* 8 (Integer/bitCount (bit-and moment-int 7))))))
       (persistent! result)))))

;;;
;;; Convenience Functions
;;;

(defn centroid
  "Compute the centroid (center of mass) of an image.
   
   Returns the geometric center weighted by pixel intensities.
   For binary images, this is the center of the foreground region.
   
   Parameters:
   - img: Input image (AFArray)
   
   Returns:
   Map containing:
   - :x - X-coordinate of centroid (double)
   - :y - Y-coordinate of centroid (double)
   - :area - Total mass/area (M00)
   
   Returns nil if image is empty (M00 = 0).
   
   Examples:
   ```clojure
   ;; Find object center
   (let [{:keys [x y area]} (centroid object-mask)]
     (println \"Object at\" [x y] \"with area\" area))
   
   ;; Track object across frames
   (doseq [frame video-frames]
     (when-let [c (centroid frame)]
       (println \"Frame centroid:\" (:x c) (:y c))))
   
   ;; Check if object present
   (if-let [c (centroid binary-img)]
     (println \"Object detected at\" [(:x c) (:y c)])
     (println \"No object found\"))
   ```"
  [^AFArray img]
  (let [{:keys [M00 M01 M10]} (moments-all img #{:m00 :m01 :m10})]
    (when (> M00 0.0)
      {:x (/ M01 M00)
       :y (/ M10 M00)
       :area M00})))

(defn area
  "Calculate the total area (pixel count or intensity sum) of an image.
   
   For binary images, this is the number of foreground pixels.
   For grayscale images, this is the sum of all pixel intensities.
   
   Parameters:
   - img: Input image (AFArray)
   
   Returns:
   Double value representing M00 (zeroth moment)
   
   Examples:
   ```clojure
   ;; Count foreground pixels
   (let [pixel-count (area binary-mask)]
     (println \"Object has\" pixel-count \"pixels\"))
   
   ;; Measure region size
   (let [size-pixels (area segmented-region)
         size-mm2 (* size-pixels pixel-spacing pixel-spacing)]
     {:pixels size-pixels
      :area-mm2 size-mm2})
   
   ;; Check minimum size threshold
   (when (> (area detected-object) 100)
     (process-object detected-object))
   ```"
  [^AFArray img]
  (:M00 (moments-all img :m00)))

(defn moments-first-order
  "Compute all first-order moments and return as map.
   
   Convenience function that computes M00, M01, M10, M11 and returns
   them as a map. Equivalent to (moments-all img :first-order).
   
   Parameters:
   - img: Input 2D image (AFArray)
   
   Returns:
   Map containing:
   - :M00 - Zeroth moment (total mass/area)
   - :M01 - First moment about y-axis
   - :M10 - First moment about x-axis
   - :M11 - Second mixed moment
   
   Examples:
   ```clojure
   ;; Get all moments
   (let [{:keys [M00 M01 M10 M11]} (moments-first-order img)]
     {:centroid [(/ M01 M00) (/ M10 M00)]
      :area M00
      :mixed-moment M11})
   
   ;; Shape analysis
   (let [m (moments-first-order shape)
         cx (/ (:M01 m) (:M00 m))
         cy (/ (:M10 m) (:M00 m))
         ;; Central moment (relative to centroid)
         mu11 (- (:M11 m) (* cx (:M10 m)))]
     {:centroid [cx cy]
      :central-moment mu11})
   ```"
  [^AFArray img]
  (moments-all img :first-order))

