(ns org.soulspace.arrayfire.ffi.rgb-gray
  "Bindings for ArrayFire RGB/Grayscale color conversion functions.
   
   This module provides functions for converting between RGB color images
   and grayscale representations. These are fundamental operations in image
   processing and computer vision.
   
   # What is RGB to Grayscale Conversion?
   
   Converting a 3-channel RGB image to a single-channel grayscale image
   involves computing a weighted combination of the color channels. The
   weights are chosen based on human perception of luminance.
   
   ## Standard Luminance Weights (ITU-R BT.601)
   
   The default conversion formula uses perceptual weights that match human
   vision sensitivity:
   
   ```
   Gray = 0.2126 * R + 0.7152 * G + 0.0722 * B
   ```
   
   These weights reflect that human eyes are most sensitive to green light,
   moderately sensitive to red, and least sensitive to blue.
   
   ## Mathematical Definition
   
   For an RGB image with dimensions [width, height, 3]:
   
   **RGB → Gray**:
   ```
   gray[x,y] = rPercent * rgb[x,y,0] + gPercent * rgb[x,y,1] + bPercent * rgb[x,y,2]
   ```
   
   **Gray → RGB**:
   ```
   rgb[x,y,0] = gray[x,y] * rFactor
   rgb[x,y,1] = gray[x,y] * gFactor
   rgb[x,y,2] = gray[x,y] * bFactor
   ```
   
   # Alternative Color Space Conversions
   
   For more sophisticated color space handling, see:
   - **HSV/RGB**: `org.soulspace.arrayfire.ffi.hsv-rgb`
   - **YCbCr/RGB**: `org.soulspace.arrayfire.ffi.ycbcr-rgb`
   - **General**: `af-color-space` function (not yet bound)
   
   # Performance Characteristics
   
   - **Time Complexity**: O(N) where N = number of pixels
   - **Space Complexity**: O(M + N) for input + output arrays
   - **GPU Acceleration**: Fully parallel across all pixels
   - **Throughput**: ~10M pixels/ms on modern GPUs
   
   The operations are highly parallelizable since each output pixel is
   computed independently.
   
   # Type Support
   
   **Input Types** (all supported):
   - Floating-point: f32, f64
   - Unsigned integers: u8, u16, u32
   - Signed integers: s8, s16, s32
   
   **Output Types**:
   - RGB → Gray: Always produces float (f32) output
   - Gray → RGB: Produces float (f32) output
   
   Integer inputs are internally converted to float for the weighted
   combination, ensuring accurate results without overflow.
   
   # Common Use Cases
   
   ## RGB to Grayscale
   
   1. **Preprocessing for algorithms**: Many computer vision algorithms
      (edge detection, feature extraction) work on grayscale images
   
   2. **Dimensionality reduction**: 3 channels → 1 channel reduces
      memory and computation by 3×
   
   3. **Display compatibility**: Some displays or printers only support
      grayscale
   
   4. **Human perception modeling**: Grayscale approximates how humans
      perceive brightness
   
   ## Grayscale to RGB
   
   1. **Visualization**: Display grayscale data on RGB displays
   
   2. **Color mapping preparation**: Create base image before applying
      color lookup tables
   
   3. **Channel replication**: Create 3-channel input for algorithms
      that expect RGB
   
   4. **Color tinting**: Use factors < 1.0 to create sepia or other
      monochromatic effects
   
   # Applications
   
   - **Medical Imaging**: X-rays, CT scans are often grayscale
   - **Infrared/Thermal Imaging**: Single-channel sensors
   - **Astronomy**: Most telescope cameras are monochrome
   - **Document Processing**: Text documents in grayscale
   - **Machine Learning**: Reduce input dimensionality for training
   - **Video Compression**: YCbCr uses grayscale-like luminance channel
   
   # Best Practices
   
   1. **Use Standard Weights**: The default ITU-R BT.601 weights
      (0.2126, 0.7152, 0.0722) are perceptually accurate for most uses
   
   2. **Preserve Dynamic Range**: Ensure weights sum to 1.0 to maintain
      brightness levels
   
   3. **Consider Gamma**: For perceptually uniform results, convert to
      linear space before weighting (most images are gamma-encoded)
   
   4. **Batch Processing**: Process multiple images in 4D arrays for
      efficiency: [width, height, channels, batch]
   
   5. **Type Conversion**: Use appropriate input types to avoid unnecessary
      conversions
   
   6. **Gray → RGB Factors**: For neutral grayscale, use (1.0, 1.0, 1.0).
      For sepia tone, try (1.0, 0.95, 0.82)
   
   # Common Pitfalls
   
   1. **Dimension Mismatch**: RGB → Gray expects 3-channel input
      (dims[2] = 3), Gray → RGB expects 2D input (dims[2] = 1)
   
   2. **Weight Normalization**: Weights not summing to 1.0 will change
      brightness levels
   
   3. **Incorrect Channel Order**: ArrayFire uses RGB order, not BGR
      (as used by some libraries like OpenCV)
   
   4. **Integer Overflow**: With integer types, ensure weighted sum
      doesn't exceed type range (use float types when uncertain)
   
   5. **Gamma Correction**: Ignoring gamma can produce perceptually
      incorrect results (images appear too dark or light)
   
   6. **Memory Leaks**: Always release arrays with `af-release-array`
      after use
   
   # Implementation Notes
   
   ## RGB to Grayscale Algorithm
   
   ```cpp
   // Extract channels as separate slices
   Array R = input[..., 0]  // Red channel
   Array G = input[..., 1]  // Green channel
   Array B = input[..., 2]  // Blue channel
   
   // Compute weighted combination
   Array gray = rPercent * R + gPercent * G + bPercent * B
   ```
   
   ## Grayscale to RGB Algorithm
   
   ```cpp
   // Multiply grayscale by factors
   Array R = gray * rFactor
   Array G = gray * gFactor
   Array B = gray * bFactor
   
   // Join into 3-channel array
   Array rgb = join(2, {R, G, B})  // Join along channel dimension
   ```
   
   ## Optimization: Equal Factors
   
   When converting Gray → RGB with factors (1.0, 1.0, 1.0), ArrayFire
   uses an optimized path:
   
   ```cpp
   // Fast path: tile grayscale along channel dimension
   Array rgb = tile(gray, [1, 1, 3, 1])
   ```
   
   This is much faster than separate multiplications.
   
   # Relationship to Other Operations
   
   - **af-color-space**: General color space conversion framework
   - **af-hsv2rgb / af-rgb2hsv**: HSV color space conversions
   - **af-ycbcr2rgb / af-rgb2ycbcr**: YCbCr conversions (separates
     luminance from chrominance)
   - **Image filters**: Often applied after grayscale conversion
   - **Edge detection**: Typically requires grayscale input
   
   # Historical Context
   
   The ITU-R BT.601 standard (formerly CCIR 601) was established for
   standard-definition television. The weights derive from the spectral
   sensitivity of the human eye's photoreceptors and the phosphors used
   in CRT displays.
   
   Modern standards like ITU-R BT.709 (HDTV) use slightly different
   weights (0.2126, 0.7152, 0.0722) due to different primary colors,
   but the difference is minimal for most applications.
   
   See also:
   - ArrayFire Image Processing: https://arrayfire.org/docs/group__image__func__rgb2gray.htm
   - ITU-R BT.601: https://en.wikipedia.org/wiki/Rec._601
   - ITU-R BT.709: https://en.wikipedia.org/wiki/Rec._709"
  (:require [coffi.ffi :as ffi :refer [defcfn]]
            [coffi.mem :as mem]
            [org.soulspace.arrayfire.ffi.loader]))

;; RGB to Grayscale conversion

;; af_err af_rgb2gray(af_array* out, const af_array in, const float rPercent, const float gPercent, const float bPercent)
(defcfn af-rgb2gray
  "Convert RGB image to grayscale using weighted combination of channels.
   
   Converts a 3-channel RGB color image to a single-channel grayscale
   image by computing a weighted sum of the color channels. The weights
   are typically chosen to match human perception of luminance.
   
   # Parameters
   
   - **out**: Pointer to output array (grayscale image)
     * Type: ::mem/pointer
     * Will contain float array with dims [width, height, 1, batch]
   
   - **in**: Input RGB image array handle
     * Type: ::mem/pointer (af_array handle)
     * Required dimensions: [width, height, 3, batch] or [width, height, 3]
     * Channel order: RGB (not BGR)
     * Supported input types: f32, f64, u8, u16, u32, s8, s16, s32
   
   - **r-percent**: Weight for red channel contribution
     * Type: ::mem/float
     * Range: Typically [0.0, 1.0]
     * Default (ITU-R BT.601): 0.2126
   
   - **g-percent**: Weight for green channel contribution
     * Type: ::mem/float
     * Range: Typically [0.0, 1.0]
     * Default (ITU-R BT.601): 0.7152
   
   - **b-percent**: Weight for blue channel contribution
     * Type: ::mem/float
     * Range: Typically [0.0, 1.0]
     * Default (ITU-R BT.601): 0.0722
   
   # Formula
   
   ```
   gray[x,y] = rPercent * rgb[x,y,0] + gPercent * rgb[x,y,1] + bPercent * rgb[x,y,2]
   ```
   
   # Dimension Behavior
   
   Input dimensions:  [W, H, 3, B]  (3 channels: R, G, B)
   Output dimensions: [W, H, 1, B]  (1 channel: grayscale)
   
   Where:
   - W = width (dimension 0)
   - H = height (dimension 1)
   - 3 = RGB channels (dimension 2)
   - B = batch size (dimension 3, optional)
   
   # Standard Weight Options
   
   ## ITU-R BT.601 (SD Video, Default)
   - Red:   0.2126 (21.26%)
   - Green: 0.7152 (71.52%)
   - Blue:  0.0722 (7.22%)
   - Use case: Standard definition video, general purpose
   
   ## ITU-R BT.709 (HD Video)
   - Red:   0.2126 (21.26%)
   - Green: 0.7152 (71.52%)
   - Blue:  0.0722 (7.22%)
   - Use case: High definition video (same as BT.601 for practical purposes)
   
   ## Equal Weights (Average)
   - Red:   0.3333 (33.33%)
   - Green: 0.3333 (33.33%)
   - Blue:  0.3333 (33.33%)
   - Use case: When perceptual accuracy not critical, simple averaging
   
   ## Lightness (Max/Min)
   - Not directly supported by this function
   - Use: (max(R,G,B) + min(R,G,B)) / 2
   - Requires separate implementation with reduction operations
   
   # Performance
   
   - **Time Complexity**: O(W * H * B)
   - **Space Complexity**: O(W * H * B) for output
   - **GPU Parallelization**: Full parallel across all pixels
   - **Throughput**: ~10-15M pixels/ms on modern GPUs
   
   For a 1920×1080 RGB image:
   - CPU: ~50-100 ms
   - GPU: ~0.2-0.5 ms
   - Speedup: 100-250×
   
   # Type Support
   
   **Input types** (all supported):
   ```
   f32, f64    → Floating-point (direct computation)
   u8, u16, u32 → Unsigned integers (converted to float)
   s8, s16, s32 → Signed integers (converted to float)
   ```
   
   **Output type**: Always float (f32) regardless of input type
   
   This ensures accurate weighted sums without overflow or precision loss.
   
   # Examples
   
   ## Example 1: Standard Grayscale Conversion (Default Weights)
   
   Convert RGB image to grayscale using ITU-R BT.601 standard weights:
   
   ```clojure
   (let [;; RGB image: 512×512 pixels, 3 channels
         rgb (af-create-array rgb-data-ptr [512 512 3])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; ITU-R BT.601 weights (perceptually accurate)
         r-weight 0.2126  ;; 21.26% red
         g-weight 0.7152  ;; 71.52% green
         b-weight 0.0722  ;; 7.22% blue
         
         result (af-rgb2gray out-ptr rgb r-weight g-weight b-weight)]
     
     (when (zero? result)
       (let [gray (mem/read-pointer out-ptr ::mem/pointer)]
         ;; gray is now [512, 512, 1] float array
         ;; Values preserve perceptual brightness
         gray)))
   ```
   
   Application: Standard image preprocessing for computer vision algorithms
   
   ## Example 2: Equal Channel Averaging
   
   Simple arithmetic mean of RGB channels (non-perceptual):
   
   ```clojure
   (let [rgb (af-create-array img-data [1920 1080 3])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Equal weights (1/3 each channel)
         equal-weight (/ 1.0 3.0)  ;; 0.3333...
         
         result (af-rgb2gray out-ptr rgb 
                            equal-weight 
                            equal-weight 
                            equal-weight)]
     
     (when (zero? result)
       ;; Output is simple average: (R + G + B) / 3
       (mem/read-pointer out-ptr ::mem/pointer)))
   ```
   
   Application: Quick luminance estimate when perceptual accuracy not critical
   
   ## Example 3: Custom Weights for Specific Application
   
   Emphasize red channel for skin tone detection:
   
   ```clojure
   (let [face-img (af-create-array img-data [640 480 3])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Custom weights emphasizing red (skin tones)
         r-weight 0.5  ;; 50% red (increased)
         g-weight 0.4  ;; 40% green
         b-weight 0.1  ;; 10% blue (decreased)
         ;; Note: Sum = 1.0 to preserve brightness
         
         result (af-rgb2gray out-ptr face-img r-weight g-weight b-weight)]
     
     (when (zero? result)
       ;; Grayscale with enhanced skin tone contrast
       (mem/read-pointer out-ptr ::mem/pointer)))
   ```
   
   Application: Medical imaging, dermatology analysis, face detection preprocessing
   
   ## Example 4: Batch Processing Multiple Images
   
   Convert batch of 10 images simultaneously:
   
   ```clojure
   (let [;; Batch: 10 images of 256×256 pixels
         rgb-batch (af-create-array batch-data [256 256 3 10])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Standard weights
         result (af-rgb2gray out-ptr rgb-batch 0.2126 0.7152 0.0722)]
     
     (when (zero? result)
       (let [gray-batch (mem/read-pointer out-ptr ::mem/pointer)]
         ;; Output: [256, 256, 1, 10]
         ;; All 10 images converted in parallel on GPU
         gray-batch)))
   ```
   
   Performance: ~10× faster than processing images sequentially
   
   ## Example 5: Preserve Only Green Channel
   
   Extract green channel (useful for vegetation analysis):
   
   ```clojure
   (let [satellite-img (af-create-array img-data [2048 2048 3])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Extract only green channel
         r-weight 0.0  ;; Ignore red
         g-weight 1.0  ;; Keep green
         b-weight 0.0  ;; Ignore blue
         
         result (af-rgb2gray out-ptr satellite-img r-weight g-weight b-weight)]
     
     (when (zero? result)
       ;; Output contains only green channel values
       (mem/read-pointer out-ptr ::mem/pointer)))
   ```
   
   Application: NDVI vegetation index, chlorophyll content analysis
   
   ## Example 6: Contrast Enhancement for Edge Detection
   
   Optimize grayscale for subsequent edge detection:
   
   ```clojure
   (let [input-img (af-create-array img-data [512 512 3])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Use standard weights
         _ (af-rgb2gray out-ptr input-img 0.2126 0.7152 0.0722)
         gray (mem/read-pointer out-ptr ::mem/pointer)
         
         ;; Now ready for edge detection
         dx-ptr (mem/alloc-pointer ::mem/pointer)
         dy-ptr (mem/alloc-pointer ::mem/pointer)
         _ (af-sobel-operator dx-ptr dy-ptr gray 3)]
     
     ;; Edge gradients computed on grayscale
     {:gradient-x (mem/read-pointer dx-ptr ::mem/pointer)
      :gradient-y (mem/read-pointer dy-ptr ::mem/pointer)})
   ```
   
   Pipeline: RGB → Grayscale → Edge Detection
   
   ## Example 7: Dimensionality Reduction for ML
   
   Reduce input size for neural network training:
   
   ```clojure
   (let [;; Training batch: 64 RGB images
         rgb-batch (af-create-array training-data [224 224 3 64])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         result (af-rgb2gray out-ptr rgb-batch 0.2126 0.7152 0.0722)]
     
     (when (zero? result)
       (let [gray-batch (mem/read-pointer out-ptr ::mem/pointer)]
         ;; Output: [224, 224, 1, 64]
         ;; 3× reduction in data size
         ;; Training speed: ~3× faster
         ;; Memory usage: ~3× less
         gray-batch)))
   ```
   
   Benefits: Faster training, less memory, often comparable accuracy
   
   ## Example 8: Document Scanning
   
   Convert color document scan to grayscale for OCR:
   
   ```clojure
   (let [;; Scanned document: high resolution
         scan-img (af-create-array scan-data [3300 2550 3])  ;; 300 DPI A4
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Standard grayscale conversion
         _ (af-rgb2gray out-ptr scan-img 0.2126 0.7152 0.0722)
         gray (mem/read-pointer out-ptr ::mem/pointer)
         
         ;; Optional: threshold for binary (black/white)
         thresh-ptr (mem/alloc-pointer ::mem/pointer)
         _ (af-threshold-binary thresh-ptr gray 0.5)]
     
     ;; Ready for OCR processing
     (mem/read-pointer thresh-ptr ::mem/pointer))
   ```
   
   Pipeline: Color Scan → Grayscale → Threshold → OCR
   
   # Validation
   
   The function performs these checks:
   
   1. **Input dimensions**: Must have exactly 3 channels (dims[2] = 3)
      - Error: AF_ERR_SIZE if dims[2] ≠ 3
   
   2. **Input not empty**: Array must have elements > 0
      - If empty (0 elements), returns empty array
   
   3. **Type support**: Input type must be supported numeric type
      - Error: AF_ERR_TYPE for unsupported types (complex, boolean)
   
   # Error Handling
   
   Returns ArrayFire error code:
   - **AF_SUCCESS (0)**: Conversion successful
   - **AF_ERR_ARG**: Invalid arguments (null pointers)
   - **AF_ERR_SIZE**: Wrong number of channels (expected 3)
   - **AF_ERR_TYPE**: Unsupported input type
   - **AF_ERR_NO_MEM**: Insufficient GPU memory
   
   # Notes
   
   1. **Perceptual Weights**: The default weights (0.2126, 0.7152, 0.0722)
      are scientifically derived from human visual perception. Don't change
      them unless you have a specific reason.
   
   2. **Weight Normalization**: Weights should typically sum to 1.0 to
      preserve overall brightness. Sums < 1.0 darken the image, > 1.0
      brighten it (with potential clipping).
   
   3. **Channel Order**: ArrayFire uses RGB order (not BGR). If your data
      is in BGR order (e.g., from OpenCV), swap r-percent and b-percent.
   
   4. **Gamma Correction**: Most images are gamma-encoded (sRGB). For
      perceptually linear results, convert to linear space first, then
      apply weights, then convert back to gamma space. (ArrayFire doesn't
      do this automatically.)
   
   5. **Output Type**: Always produces float output, even for integer
      inputs. This ensures no precision loss from weighted summation.
   
   6. **Batch Processing**: Input can be 4D [W, H, 3, B] for batch
      processing. All images converted in parallel on GPU.
   
   7. **Memory Management**: Caller must release both input and output
      arrays with `af-release-array` to avoid memory leaks.
   
   # Returns
   
   ArrayFire error code (af_err enum):
   - **0 (AF_SUCCESS)**: Conversion completed successfully
   - **Non-zero**: Error occurred (see error codes above)
   
   On success, the output pointer is updated with the grayscale array handle.
   
   # See Also
   
   - `af-gray2rgb`: Convert grayscale to RGB (inverse operation)
   - `af-color-space`: General color space conversion framework
   - `af-rgb2hsv`: RGB to HSV conversion (for hue-based processing)
   - `af-rgb2ycbcr`: RGB to YCbCr (separates luminance from chrominance)"
  "af_rgb2gray" [::mem/pointer ::mem/pointer ::mem/float ::mem/float ::mem/float] ::mem/int)

;; Grayscale to RGB conversion

;; af_err af_gray2rgb(af_array* out, const af_array in, const float rFactor, const float gFactor, const float bFactor)
(defcfn af-gray2rgb
  "Convert grayscale image to RGB by replicating with channel factors.
   
   Converts a single-channel grayscale image to a 3-channel RGB image by
   multiplying the grayscale values by separate factors for each channel.
   With factors (1.0, 1.0, 1.0), this creates a neutral grayscale displayed
   in RGB. Other factor combinations can create tinted effects.
   
   # Parameters
   
   - **out**: Pointer to output array (RGB image)
     * Type: ::mem/pointer
     * Will contain float array with dims [width, height, 3, batch]
   
   - **in**: Input grayscale image array handle
     * Type: ::mem/pointer (af_array handle)
     * Required dimensions: [width, height, 1, batch] or [width, height]
     * Must be 2D or 3D with single channel (dims[2] = 1)
     * Supported input types: f32, f64, u8, u16, u32, s8, s16, s32
   
   - **r-factor**: Multiplication factor for red channel
     * Type: ::mem/float
     * Range: Typically [0.0, 1.0] but any float allowed
     * Default (neutral): 1.0
   
   - **g-factor**: Multiplication factor for green channel
     * Type: ::mem/float
     * Range: Typically [0.0, 1.0] but any float allowed
     * Default (neutral): 1.0
   
   - **b-factor**: Multiplication factor for blue channel
     * Type: ::mem/float
     * Range: Typically [0.0, 1.0] but any float allowed
     * Default (neutral): 1.0
   
   # Formula
   
   ```
   rgb[x,y,0] = gray[x,y] * rFactor  // Red channel
   rgb[x,y,1] = gray[x,y] * gFactor  // Green channel
   rgb[x,y,2] = gray[x,y] * bFactor  // Blue channel
   ```
   
   # Dimension Behavior
   
   Input dimensions:  [W, H, 1, B]  or  [W, H]     (1 channel: grayscale)
   Output dimensions: [W, H, 3, B]  or  [W, H, 3]  (3 channels: R, G, B)
   
   Where:
   - W = width (dimension 0)
   - H = height (dimension 1)
   - 1 = grayscale channel (dimension 2)
   - 3 = RGB channels (dimension 2 in output)
   - B = batch size (dimension 3, optional)
   
   # Common Factor Combinations
   
   ## Neutral Grayscale (Default)
   - Red:   1.0 (100%)
   - Green: 1.0 (100%)
   - Blue:  1.0 (100%)
   - Result: Pure grayscale displayed in RGB
   - Use case: Display grayscale on RGB monitors
   
   ## Sepia Tone (Vintage Photography)
   - Red:   1.0   (100% - warm highlights)
   - Green: 0.95  (95%  - slightly reduced)
   - Blue:  0.82  (82%  - cool shadows)
   - Result: Warm, vintage brown tone
   - Use case: Artistic effects, vintage photos
   
   ## Cool Blue Tone (Moonlight Effect)
   - Red:   0.7  (70%  - reduced warmth)
   - Green: 0.8  (80%  - slight reduction)
   - Blue:  1.0  (100% - full blue)
   - Result: Cool, blue-tinted image
   - Use case: Night scenes, cold atmosphere
   
   ## Warm Red Tone (Sunset/Fire)
   - Red:   1.0  (100% - full red)
   - Green: 0.6  (60%  - reduced green)
   - Blue:  0.4  (40%  - minimal blue)
   - Result: Warm, reddish tone
   - Use case: Fire, sunset, warm effects
   
   ## Green Tone (Night Vision)
   - Red:   0.0  (0%   - no red)
   - Green: 1.0  (100% - full green)
   - Blue:  0.0  (0%   - no blue)
   - Result: Pure green channel
   - Use case: Night vision simulation
   
   # Performance
   
   - **Time Complexity**: O(W * H * B)
   - **Space Complexity**: O(3 * W * H * B) for output (3× input size)
   - **GPU Parallelization**: Full parallel across all pixels
   - **Throughput**: ~15-20M pixels/ms on modern GPUs
   
   For a 1920×1080 grayscale image:
   - CPU: ~30-60 ms
   - GPU: ~0.15-0.3 ms
   - Speedup: 100-200×
   
   ## Optimization: Equal Factors (1.0, 1.0, 1.0)
   
   When all factors equal 1.0, ArrayFire uses a fast path:
   
   ```cpp
   // Fast: tile operation (no multiplication)
   rgb = tile(gray, [1, 1, 3, 1])
   ```
   
   This is ~3× faster than the general case with multiplications.
   
   # Type Support
   
   **Input types** (all supported):
   ```
   f32, f64    → Floating-point (direct computation)
   u8, u16, u32 → Unsigned integers (converted to float)
   s8, s16, s32 → Signed integers (converted to float)
   ```
   
   **Output type**: Always float (f32) regardless of input type
   
   This ensures accurate multiplication without overflow.
   
   # Examples
   
   ## Example 1: Neutral Grayscale Display
   
   Convert grayscale to RGB for display on RGB monitor:
   
   ```clojure
   (let [;; Grayscale image: 512×512 pixels
         gray (af-create-array gray-data-ptr [512 512])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Neutral factors (no tinting)
         r-factor 1.0
         g-factor 1.0
         b-factor 1.0
         
         result (af-gray2rgb out-ptr gray r-factor g-factor b-factor)]
     
     (when (zero? result)
       (let [rgb (mem/read-pointer out-ptr ::mem/pointer)]
         ;; rgb is [512, 512, 3] float array
         ;; All channels identical, looks grayscale
         rgb)))
   ```
   
   Application: Display grayscale images on RGB displays
   
   ## Example 2: Sepia Tone Effect
   
   Create vintage sepia-toned photograph:
   
   ```clojure
   (let [gray (af-create-array img-data [1920 1080])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Sepia factors (warm brown tone)
         r-factor 1.0   ;; Full red
         g-factor 0.95  ;; Slight reduction
         b-factor 0.82  ;; Significant reduction
         
         result (af-gray2rgb out-ptr gray r-factor g-factor b-factor)]
     
     (when (zero? result)
       ;; Output has vintage brown tone
       (mem/read-pointer out-ptr ::mem/pointer)))
   ```
   
   Application: Artistic photography, vintage effects, Instagram-style filters
   
   ## Example 3: Create Heatmap Base
   
   Prepare grayscale for colormap application:
   
   ```clojure
   (let [;; Thermal data as grayscale
         thermal (af-create-array thermal-data [640 480])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Convert to RGB (neutral, for now)
         _ (af-gray2rgb out-ptr thermal 1.0 1.0 1.0)
         rgb-base (mem/read-pointer out-ptr ::mem/pointer)
         
         ;; Now apply colormap (not shown)
         ;; e.g., map low values to blue, high to red
         ]
     
     ;; RGB base ready for heatmap coloring
     rgb-base)
   ```
   
   Pipeline: Thermal Data → Grayscale → RGB → Heatmap Colormap
   
   ## Example 4: Night Vision Effect
   
   Simulate night vision goggles (green monochrome):
   
   ```clojure
   (let [gray (af-create-array img-data [800 600])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Pure green channel (night vision)
         r-factor 0.0  ;; No red
         g-factor 1.0  ;; Full green
         b-factor 0.0  ;; No blue
         
         result (af-gray2rgb out-ptr gray r-factor g-factor b-factor)]
     
     (when (zero? result)
       ;; Output is pure green, like night vision
       (mem/read-pointer out-ptr ::mem/pointer)))
   ```
   
   Application: Night vision simulation, thermal imaging display, military applications
   
   ## Example 5: Channel Amplification
   
   Amplify intensity by using factors > 1.0:
   
   ```clojure
   (let [gray (af-create-array dim-img-data [512 512])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Amplify all channels by 1.5×
         amplify-factor 1.5
         
         result (af-gray2rgb out-ptr gray 
                            amplify-factor 
                            amplify-factor 
                            amplify-factor)]
     
     (when (zero? result)
       ;; Output is 50% brighter (may clip at 1.0)
       (mem/read-pointer out-ptr ::mem/pointer)))
   ```
   
   Note: Values may be clipped if they exceed valid range [0,1] or [0,255]
   
   ## Example 6: Prepare for CNN Input
   
   Convert grayscale to 3-channel for neural networks expecting RGB:
   
   ```clojure
   (let [;; Grayscale MNIST digit: 28×28
         digit (af-create-array digit-data [28 28])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Replicate to 3 channels (neutral)
         result (af-gray2rgb out-ptr digit 1.0 1.0 1.0)]
     
     (when (zero? result)
       (let [rgb-digit (mem/read-pointer out-ptr ::mem/pointer)]
         ;; Now [28, 28, 3] - compatible with RGB-trained models
         ;; e.g., transfer learning from ImageNet
         rgb-digit)))
   ```
   
   Use case: Apply RGB-trained models to grayscale data
   
   ## Example 7: Batch Processing with Tinting
   
   Apply sepia tone to batch of images:
   
   ```clojure
   (let [;; Batch: 20 grayscale images
         gray-batch (af-create-array batch-data [256 256 1 20])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Sepia factors
         result (af-gray2rgb out-ptr gray-batch 1.0 0.95 0.82)]
     
     (when (zero? result)
       (let [rgb-batch (mem/read-pointer out-ptr ::mem/pointer)]
         ;; Output: [256, 256, 3, 20]
         ;; All 20 images tinted in parallel
         rgb-batch)))
   ```
   
   Performance: ~10× faster than sequential processing
   
   ## Example 8: Duotone Effect
   
   Create custom duotone (two-color gradient):
   
   ```clojure
   (let [gray (af-create-array img-data [1024 768])
         out-ptr (mem/alloc-pointer ::mem/pointer)
         
         ;; Duotone: shadows blue, highlights orange
         ;; First create RGB base
         _ (af-gray2rgb out-ptr gray 1.0 0.7 0.4)]
     
     ;; Output has orange/blue duotone effect
     ;; (Simplified - full duotone needs more processing)
     (mem/read-pointer out-ptr ::mem/pointer))
   ```
   
   Application: Artistic effects, poster design, Spotify-style duotones
   
   # Validation
   
   The function performs these checks:
   
   1. **Input dimensions**: Must have 1 channel (dims[2] = 1) or be 2D
      - Error: AF_ERR_SIZE if dims[2] ≠ 1 (for 3D arrays)
   
   2. **Input not empty**: Array must have elements > 0
      - If empty (0 elements), returns empty array
   
   3. **Type support**: Input type must be supported numeric type
      - Error: AF_ERR_TYPE for unsupported types (complex, boolean)
   
   # Error Handling
   
   Returns ArrayFire error code:
   - **AF_SUCCESS (0)**: Conversion successful
   - **AF_ERR_ARG**: Invalid arguments (null pointers)
   - **AF_ERR_SIZE**: Wrong number of channels (expected 1)
   - **AF_ERR_TYPE**: Unsupported input type
   - **AF_ERR_NO_MEM**: Insufficient GPU memory
   
   # Notes
   
   1. **Not Inverse of rgb2gray**: This is NOT the inverse operation of
      `af-rgb2gray`. Once an image is converted to grayscale, the original
      color information is lost and cannot be recovered. This function
      only creates a colored version of grayscale data.
   
   2. **Factor Interpretation**: Factors multiply the grayscale intensity:
      - 1.0 = preserve original intensity
      - < 1.0 = darken that channel
      - > 1.0 = brighten that channel (may clip)
   
   3. **Equal Factors Optimization**: When r-factor = g-factor = b-factor = 1.0,
      ArrayFire uses an optimized tiling operation (~3× faster).
   
   4. **Output Type**: Always produces float output for accuracy, even
      with integer inputs.
   
   5. **Clipping**: Values outside the valid range may be clipped by
      subsequent operations. ArrayFire typically uses [0,1] for float
      images, [0,255] for byte images.
   
   6. **Batch Processing**: Input can be 3D [W, H, 1] or 4D [W, H, 1, B]
      for batch processing.
   
   7. **Memory Management**: Caller must release both input and output
      arrays with `af-release-array`.
   
   8. **Alternative Colorization**: For true image colorization (adding
      color to grayscale), you need machine learning models or manual
      selection. This function only provides uniform tinting.
   
   # Returns
   
   ArrayFire error code (af_err enum):
   - **0 (AF_SUCCESS)**: Conversion completed successfully
   - **Non-zero**: Error occurred (see error codes above)
   
   On success, the output pointer is updated with the RGB array handle.
   
   # See Also
   
   - `af-rgb2gray`: Convert RGB to grayscale (reduction operation)
   - `af-color-space`: General color space conversion framework
   - `af-hsv2rgb`: HSV to RGB conversion (for hue-based colorization)
   - `af-tile`: Replicate array along dimensions (used internally for equal factors)"
  "af_gray2rgb" [::mem/pointer ::mem/pointer ::mem/float ::mem/float ::mem/float] ::mem/int)
