(ns org.soulspace.arrayfire.ffi.hsv-rgb
  "Bindings for the ArrayFire HSV/RGB color space conversion functions.
   
   ## Overview
   
   This namespace provides FFI bindings for ArrayFire's HSV-RGB color space
   conversion functions. These functions enable bidirectional conversion
   between RGB (Red-Green-Blue) and HSV (Hue-Saturation-Value) color spaces,
   which are fundamental operations in image processing, computer vision, and
   graphics applications.
   
   ## Color Space Fundamentals
   
   ### RGB Color Space
   
   RGB is an additive color model where colors are represented as combinations
   of red, green, and blue light intensities. It's the most common color space
   for digital images and displays.
   
   **Components:**
   - **R (Red)**: Intensity of red component, range [0, 1]
   - **G (Green)**: Intensity of green component, range [0, 1]
   - **B (Blue)**: Intensity of blue component, range [0, 1]
   
   **Characteristics:**
   - Device-dependent (different displays may show different colors)
   - Natural for hardware (monitors, cameras)
   - Not perceptually uniform (equal numeric differences ≠ equal perceived differences)
   - Difficult to adjust color properties (hue, saturation, brightness) independently
   
   **Use Cases:**
   - Image capture and display
   - Web graphics and video
   - Raw sensor data processing
   
   ### HSV Color Space
   
   HSV (also called HSB - Hue-Saturation-Brightness) is a cylindrical color
   model that represents colors in terms more aligned with human perception.
   
   **Components:**
   - **H (Hue)**: Color type/angle on color wheel, range [0, 1] (maps to 0°-360°)
     * 0.0 = Red (0°)
     * 0.167 = Yellow (60°)
     * 0.333 = Green (120°)
     * 0.5 = Cyan (180°)
     * 0.667 = Blue (240°)
     * 0.833 = Magenta (300°)
     * 1.0 = Red (360° = 0°)
   
   - **S (Saturation)**: Color intensity/purity, range [0, 1]
     * 0.0 = Grayscale (no color)
     * 1.0 = Pure/vivid color
   
   - **V (Value)**: Brightness, range [0, 1]
     * 0.0 = Black
     * 1.0 = Maximum brightness
   
   **Characteristics:**
   - Perceptually intuitive (separates chromaticity from brightness)
   - Easy to manipulate color properties independently
   - Natural for color selection interfaces
   - Useful for color-based image segmentation
   
   **Geometric Interpretation:**
   - Hue: Angle around the central axis (color wheel)
   - Saturation: Distance from the central axis (color purity)
   - Value: Height along the central axis (brightness)
   
   ## Conversion Formulas
   
   ### RGB to HSV
   
   Given RGB values (R, G, B) in range [0, 1]:
   
   ```
   Cmax = max(R, G, B)
   Cmin = min(R, G, B)
   Δ = Cmax - Cmin
   
   Hue (H):
     If Δ = 0:          H = 0
     If Cmax = R:       H = ((G - B) / Δ) mod 6
     If Cmax = G:       H = ((B - R) / Δ) + 2
     If Cmax = B:       H = ((R - G) / Δ) + 4
     H = H / 6  (normalize to [0, 1])
   
   Saturation (S):
     If Cmax = 0:       S = 0
     Otherwise:         S = Δ / Cmax
   
   Value (V):
     V = Cmax
   ```
   
   **Properties:**
   - Undefined hue when saturation is zero (achromatic/grayscale)
   - Hue is periodic (wraps around at 0/1)
   - Conversion is fast (no transcendental functions)
   
   ### HSV to RGB
   
   Given HSV values (H, S, V) in range [0, 1]:
   
   ```
   C = V × S  (chroma)
   H' = H × 6  (hue sector)
   X = C × (1 - |H' mod 2 - 1|)
   m = V - C
   
   Depending on H' sector [0, 6):
     [0, 1): (R', G', B') = (C, X, 0)
     [1, 2): (R', G', B') = (X, C, 0)
     [2, 3): (R', G', B') = (0, C, X)
     [3, 4): (R', G', B') = (0, X, C)
     [4, 5): (R', G', B') = (X, 0, C)
     [5, 6): (R', G', B') = (C, 0, X)
   
   (R, G, B) = (R' + m, G' + m, B' + m)
   ```
   
   **Properties:**
   - Handles all cases including edge values
   - Efficient computation (conditional branches)
   - Exact round-trip conversion (within floating-point precision)
   
   ## Implementation Details
   
   ### Array Format Requirements
   
   Both conversion functions expect 3-dimensional arrays where:
   - **Dimension 0**: Width (columns)
   - **Dimension 1**: Height (rows)
   - **Dimension 2**: Channels (must be 3)
   - **Dimension 3**: Batch dimension (optional, for multiple images)
   
   **Channel Ordering:**
   - RGB: Channel 0 = Red, Channel 1 = Green, Channel 2 = Blue
   - HSV: Channel 0 = Hue, Channel 1 = Saturation, Channel 2 = Value
   
   **Value Ranges:**
   All components must be in the range [0, 1]. Values outside this range
   will produce undefined results.
   
   ### GPU Acceleration
   
   ArrayFire's HSV-RGB conversions are fully GPU-accelerated:
   
   **Parallelization Strategy:**
   - Each pixel processed independently (embarrassingly parallel)
   - No inter-pixel dependencies
   - Coalesced memory access patterns
   - Efficient SIMD/vectorization
   
   **Performance (typical):**
   - 640×480 image: 0.1-0.5ms on GPU vs 5-20ms on CPU
   - 1920×1080 image: 0.3-1.5ms on GPU vs 20-80ms on CPU
   - 4K (3840×2160): 1-5ms on GPU vs 80-300ms on CPU
   - Speedup: 20-100× depending on image size
   
   ### Numerical Precision
   
   **Floating-Point Considerations:**
   - Both f32 and f64 supported
   - f32 sufficient for most applications (error ~1e-6)
   - f64 recommended for scientific applications requiring high precision
   
   **Edge Cases:**
   - Grayscale colors (S=0): Hue undefined, set to 0
   - Black (V=0): Both H and S undefined, set to 0
   - Pure colors: Handled exactly (no precision loss)
   
   **Round-Trip Accuracy:**
   - RGB → HSV → RGB: Typical error <1e-6 (f32) or <1e-15 (f64)
   - HSV → RGB → HSV: Exact for most values, hue may differ for grayscale
   
   ## Function: af-hsv2rgb
   
   Convert an image from HSV (Hue-Saturation-Value) color space to RGB
   (Red-Green-Blue) color space.
   
   ### Parameters
   
   - **out** (out): `::mem/pointer`
     Pointer to output array handle (af_array) for RGB image.
     The output array will have the same dimensions and type as input.
     Channel ordering: [R, G, B] where:
     - R (Red): First channel
     - G (Green): Second channel
     - B (Blue): Third channel
     All values in range [0, 1].
   
   - **in** (in): `::mem/pointer`
     Input array handle (af_array) for HSV image.
     Must be 3-dimensional with 3 channels.
     Channel ordering: [H, S, V] where:
     - H (Hue): First channel, range [0, 1] (0° to 360°)
     - S (Saturation): Second channel, range [0, 1]
     - V (Value): Third channel, range [0, 1]
     Supported types: f32, f64.
   
   ### Returns
   
   `::mem/int` - ArrayFire error code:
   - `0` (AF_SUCCESS): Conversion successful
   - `201` (AF_ERR_SIZE): Input array not 3-dimensional or channels ≠ 3
   - `202` (AF_ERR_TYPE): Input array not f32 or f64 type
   - `205` (AF_ERR_NOT_SUPPORTED): Backend doesn't support conversion (e.g., oneAPI)
   
   ### Algorithm
   
   For each pixel (H, S, V):
   1. Compute chroma: C = V × S
   2. Determine hue sector: H' = H × 6, sector = floor(H')
   3. Compute intermediate: X = C × (1 - |H' mod 2 - 1|)
   4. Match component: m = V - C
   5. Select RGB' based on sector:
      - [0, 1): (C, X, 0)
      - [1, 2): (X, C, 0)
      - [2, 3): (0, C, X)
      - [3, 4): (0, X, C)
      - [4, 5): (X, 0, C)
      - [5, 6): (C, 0, X)
   6. Add match component: (R, G, B) = (R', G', B') + m
   
   **Complexity:** O(N) where N = number of pixels
   
   ## Function: af-rgb2hsv
   
   Convert an image from RGB (Red-Green-Blue) color space to HSV
   (Hue-Saturation-Value) color space.
   
   ### Parameters
   
   - **out** (out): `::mem/pointer`
     Pointer to output array handle (af_array) for HSV image.
     The output array will have the same dimensions and type as input.
     Channel ordering: [H, S, V] where:
     - H (Hue): First channel, range [0, 1] (maps to 0° to 360°)
     - S (Saturation): Second channel, range [0, 1]
     - V (Value): Third channel, range [0, 1]
     All values in range [0, 1].
   
   - **in** (in): `::mem/pointer`
     Input array handle (af_array) for RGB image.
     Must be 3-dimensional with 3 channels.
     Channel ordering: [R, G, B] where:
     - R (Red): First channel, range [0, 1]
     - G (Green): Second channel, range [0, 1]
     - B (Blue): Third channel, range [0, 1]
     Supported types: f32, f64.
   
   ### Returns
   
   `::mem/int` - ArrayFire error code:
   - `0` (AF_SUCCESS): Conversion successful
   - `201` (AF_ERR_SIZE): Input array not 3-dimensional or channels ≠ 3
   - `202` (AF_ERR_TYPE): Input array not f32 or f64 type
   - `205` (AF_ERR_NOT_SUPPORTED): Backend doesn't support conversion (e.g., oneAPI)
   
   ### Algorithm
   
   For each pixel (R, G, B):
   1. Compute max and min: Cmax = max(R, G, B), Cmin = min(R, G, B)
   2. Compute delta: Δ = Cmax - Cmin
   3. Compute hue H:
      - If Δ = 0: H = 0 (undefined, grayscale)
      - If Cmax = R: H = ((G - B) / Δ) mod 6
      - If Cmax = G: H = ((B - R) / Δ) + 2
      - If Cmax = B: H = ((R - G) / Δ) + 4
      - Normalize: H = H / 6
   4. Compute saturation S:
      - If Cmax = 0: S = 0
      - Otherwise: S = Δ / Cmax
   5. Compute value: V = Cmax
   
   **Complexity:** O(N) where N = number of pixels
   
   ## Use Cases
   
   ### 1. Color-Based Object Detection and Tracking
   
   HSV is ideal for detecting objects by color because hue is independent of
   lighting conditions (unlike RGB where all channels change with lighting).
   
   **Workflow:**
   1. Convert camera frame from RGB to HSV
   2. Define hue range for target color (e.g., red: [0, 0.05] ∪ [0.95, 1])
   3. Create binary mask where hue falls in range
   4. Apply morphological operations to clean mask
   5. Find contours and track objects
   
   **Example Scenario:**
   - Application: Autonomous robot tracking colored ball
   - Input: 640×480 RGB video at 30fps
   - Target: Orange ball (hue ≈ 0.05-0.08)
   - Processing: RGB→HSV (0.5ms), thresholding (0.1ms), morphology (0.3ms)
   - Total: <1ms per frame, real-time at 30fps
   
   **Parameters:**
   - Hue range: [0.04, 0.09] (orange)
   - Saturation range: [0.5, 1.0] (exclude pale/washed out)
   - Value range: [0.3, 1.0] (exclude dark/shadowed)
   
   **Advantages:**
   - Robust to lighting variations (shadows, highlights)
   - Simple threshold definition (single hue range)
   - Fast processing (GPU-accelerated conversions)
   
   ### 2. Image Enhancement and Color Adjustment
   
   HSV enables intuitive manipulation of image properties by adjusting
   individual channels independently.
   
   **Common Adjustments:**
   
   **a) Brightness Enhancement:**
   - Convert RGB → HSV
   - Multiply V channel by factor (e.g., 1.2 for 20% brighter)
   - Clamp to [0, 1]
   - Convert HSV → RGB
   
   **b) Saturation Boost:**
   - Convert RGB → HSV
   - Multiply S channel by factor (e.g., 1.5 for 50% more saturated)
   - Clamp to [0, 1]
   - Convert HSV → RGB
   
   **c) Hue Shift (Color Tinting):**
   - Convert RGB → HSV
   - Add offset to H channel (e.g., +0.1 for blue tint)
   - Wrap modulo 1.0
   - Convert HSV → RGB
   
   **Example Scenario:**
   - Application: Photo editing software
   - Input: 12MP photo (4000×3000), 36 million pixels
   - Operations: Brightness +15%, Saturation +25%
   - Performance: RGB→HSV (3ms), adjustments (1ms), HSV→RGB (3ms)
   - Total: 7ms, interactive response
   
   **Quality Considerations:**
   - Use f64 for high-quality professional editing
   - Avoid extreme adjustments (>2× multiplication)
   - Preview before committing
   
   ### 3. White Balance Correction
   
   Adjust color temperature and tint by manipulating hue and saturation.
   
   **Workflow:**
   1. Convert image RGB → HSV
   2. Identify white/gray reference points
   3. Calculate hue shift needed to neutralize color cast
   4. Apply global hue adjustment
   5. Optionally adjust saturation to compensate
   6. Convert HSV → RGB
   
   **Example Scenario:**
   - Application: Correct indoor tungsten lighting (yellowish cast)
   - Input: 1920×1080 photo with warm color cast
   - Hue shift: -0.05 (shift away from yellow toward blue)
   - Saturation: ×0.9 (slightly reduce to avoid over-saturation)
   
   **Advanced Techniques:**
   - Adaptive white balance: Different corrections per region
   - Gray world assumption: Adjust until average hue is neutral
   - Reference-based: Use known neutral object for calibration
   
   ### 4. Color Segmentation for Computer Vision
   
   Segment image into regions based on color similarity in HSV space.
   
   **Workflow:**
   1. Convert RGB → HSV
   2. Define color ranges for each segment class
   3. Create binary masks for each class
   4. Optional: Refine with morphological operations
   5. Label connected components
   6. Extract features from each segment
   
   **Example Scenario:**
   - Application: Traffic sign detection
   - Target colors:
     * Red signs: H ∈ [0, 0.05] ∪ [0.95, 1], S > 0.5, V > 0.3
     * Blue signs: H ∈ [0.55, 0.65], S > 0.4, V > 0.3
     * Yellow signs: H ∈ [0.12, 0.18], S > 0.5, V > 0.5
   - Performance: 1920×1080 at 30fps, 2-3ms per frame
   
   **Advantages:**
   - Simple definition of color classes
   - Robust to illumination changes
   - Fast processing with GPU acceleration
   
   ### 5. Artistic Effects and Filters
   
   Create stylized images by selectively modifying HSV channels.
   
   **Effect Examples:**
   
   **a) Selective Colorization:**
   - Convert to HSV
   - Detect specific hue range (e.g., red)
   - Set S=0 (desaturate) for all other hues
   - Convert back to RGB
   - Result: Only red objects in color, rest grayscale
   
   **b) Complementary Color Inversion:**
   - Convert to HSV
   - Add 0.5 to H channel (shift to opposite hue)
   - Convert back to RGB
   - Result: Surreal color palette
   
   **c) Psychedelic Effect:**
   - Convert to HSV
   - Apply sine wave to H channel: H' = (H + 0.5 × sin(2π × V)) mod 1
   - Boost S channel: S' = min(1.5 × S, 1)
   - Convert back to RGB
   
   **Example Scenario:**
   - Application: Real-time video effect for VJ performance
   - Input: 1280×720 video at 60fps
   - Effect: Selective colorization (keep skin tones, desaturate background)
   - Performance: <16ms per frame (60fps), GPU-accelerated
   
   ### 6. Lighting-Invariant Feature Extraction
   
   Use hue and saturation for features robust to illumination changes.
   
   **Workflow:**
   1. Convert RGB → HSV
   2. Use H and S channels for color features
   3. Use V channel separately for texture/intensity features
   4. Combine features weighted by application needs
   
   **Example Scenario:**
   - Application: Face recognition under varying lighting
   - Input: Face images, different illumination conditions
   - Features: Hue histogram (16 bins), Saturation histogram (16 bins)
   - Robustness: Lighting changes affect V primarily, H/S more stable
   
   **Advantages:**
   - Hue largely invariant to intensity changes
   - Saturation more stable than RGB color differences
   - Better generalization across lighting conditions
   
   ## Advanced Techniques
   
   ### Vectorized Batch Processing
   
   Process multiple images in parallel using 4th dimension:
   
   ```clojure
   (let [rgb-batch (af-load-images image-paths)  ; Load as 4D array
         hsv-batch (af-rgb2hsv rgb-batch)         ; Convert entire batch
         ;; Adjust all images simultaneously
         hsv-adjusted (af-mul hsv-batch adjustment-array)
         rgb-result (af-hsv2rgb hsv-adjusted)]    ; Convert back
     rgb-result)
   ```
   
   **Performance:**
   - 100 images, 640×480 each
   - Batch conversion: 8ms total = 0.08ms per image
   - Sequential conversion: 50ms total = 0.5ms per image
   - Speedup: 6× from better GPU utilization
   
   ### Perceptual Color Distance
   
   Compute color similarity more aligned with human perception:
   
   ```clojure
   (defn color-distance-hsv
     [hsv1 hsv2]
     (let [h1 (get-channel hsv1 0)
           h2 (get-channel hsv2 0)
           s1 (get-channel hsv1 1)
           s2 (get-channel hsv2 1)
           v1 (get-channel hsv1 2)
           v2 (get-channel hsv2 2)
           ;; Circular hue distance
           dh (min (abs (- h1 h2))
                   (- 1 (abs (- h1 h2))))
           ds (abs (- s1 s2))
           dv (abs (- v1 v2))
           ;; Weighted Euclidean distance
           weights [2.0 1.0 1.0]]  ; Hue most important
       (sqrt (+ (* 2 dh dh)
                (* 1 ds ds)
                (* 1 dv dv)))))
   ```
   
   **Applications:**
   - Color-based nearest neighbor search
   - Image clustering by dominant colors
   - Color palette generation
   
   ### Adaptive Color Enhancement
   
   Enhance images based on content analysis:
   
   ```clojure
   (defn adaptive-enhance
     [rgb-image]
     (let [hsv (af-rgb2hsv rgb-image)
           h (get-channel hsv 0)
           s (get-channel hsv 1)
           v (get-channel hsv 2)
           ;; Analyze image statistics
           s-mean (af-mean-all s)
           v-mean (af-mean-all v)
           ;; Compute adaptive adjustments
           s-factor (if (< s-mean 0.3) 1.5 1.0)  ; Boost if undersaturated
           v-factor (/ 0.5 v-mean)                ; Normalize brightness
           ;; Apply adjustments
           s' (af-clamp (af-mul s s-factor) 0 1)
           v' (af-clamp (af-mul v v-factor) 0 1)
           hsv' (af-join-channels h s' v')]
       (af-hsv2rgb hsv')))
   ```
   
   ### Color Transfer Between Images
   
   Match color palette of one image to another:
   
   ```clojure
   (defn color-transfer
     [source-rgb target-rgb]
     (let [source-hsv (af-rgb2hsv source-rgb)
           target-hsv (af-rgb2hsv target-rgb)
           ;; Extract channels
           [hs ss vs] (split-channels source-hsv)
           [ht st vt] (split-channels target-hsv)
           ;; Match statistics (mean and std)
           hs-mean (af-mean-all hs)
           hs-std (af-stdev-all hs)
           ht-mean (af-mean-all ht)
           ht-std (af-stdev-all ht)
           ;; Transfer statistics
           hs' (af-add (af-mul (af-sub hs hs-mean)
                               (/ ht-std hs-std))
                       ht-mean)
           ;; Similar for S and V channels
           hsv' (af-join-channels hs' ss vs)]
       (af-hsv2rgb hsv')))
   ```
   
   **Applications:**
   - Style transfer (color palette matching)
   - Mood adjustment (make sunset-like, overcast-like)
   - Consistent look across photo series
   
   ## Performance Optimization
   
   ### Memory Layout
   
   ArrayFire uses column-major order for optimal GPU performance:
   - Consecutive pixels in dimension 0 should be processed together
   - Channel data interleaved for coalesced memory access
   
   ### Pipeline Optimization
   
   Minimize conversions by batching operations in HSV space:
   
   **Inefficient:**
   ```clojure
   (-> rgb
       (af-rgb2hsv)        ; Convert
       (adjust-brightness)
       (af-hsv2rgb)        ; Convert back
       (af-rgb2hsv)        ; Convert again!
       (adjust-saturation)
       (af-hsv2rgb))       ; Final convert
   ```
   
   **Optimized:**
   ```clojure
   (-> rgb
       (af-rgb2hsv)        ; Convert once
       (adjust-brightness)
       (adjust-saturation) ; Do all operations in HSV
       (af-hsv2rgb))       ; Convert back once
   ```
   
   ### Benchmark Results (typical)
   
   | Image Size | RGB→HSV (GPU) | HSV→RGB (GPU) | RGB→HSV (CPU) | HSV→RGB (CPU) |
   |------------|---------------|---------------|---------------|---------------|
   | 640×480    | 0.3 ms        | 0.3 ms        | 8 ms          | 8 ms          |
   | 1280×720   | 0.5 ms        | 0.5 ms        | 18 ms         | 18 ms         |
   | 1920×1080  | 1.0 ms        | 1.0 ms        | 40 ms         | 40 ms         |
   | 3840×2160  | 3.0 ms        | 3.0 ms        | 160 ms        | 160 ms        |
   
   **GPU Speedup:** 25-50× compared to CPU
   
   ## Error Handling
   
   ### Common Errors
   
   1. **AF_ERR_SIZE (201)**: Wrong dimensions
      - Input must be 3D with exactly 3 channels
      - Check: `(= (af-get-dims in) [width height 3 1])`
   
   2. **AF_ERR_TYPE (202)**: Wrong data type
      - Only f32 and f64 supported
      - Convert if needed: `(af-cast in AF_F32)`
   
   3. **AF_ERR_NOT_SUPPORTED (205)**: Backend not supported
      - oneAPI backend doesn't support HSV conversions
      - Use CPU, CUDA, or OpenCL backends
   
   ### Debugging Strategies
   
   1. **Verify input range:**
      ```clojure
      (let [min-val (af-min-all in)
            max-val (af-max-all in)]
        (when (or (< min-val 0) (> max-val 1))
          (println \"Warning: Values outside [0,1] range\")))
      ```
   
   2. **Check for NaN/Inf:**
      ```clojure
      (when (af-any-true (af-isnan in))
        (println \"Error: NaN values detected\"))
      ```
   
   3. **Visualize channels separately:**
      ```clojure
      (let [hsv (af-rgb2hsv rgb)
            h (af-get-channel hsv 0)
            s (af-get-channel hsv 1)
            v (af-get-channel hsv 2)]
        (af-display-image h \"Hue\")
        (af-display-image s \"Saturation\")
        (af-display-image v \"Value\"))
      ```
   
   ## Integration Patterns
   
   ### Color-Based Object Tracking
   
   ```clojure
   (defn track-colored-object
     [rgb-frame target-hue tolerance]
     (let [hsv (af-rgb2hsv rgb-frame)
           h (af-get-channel hsv 0)
           s (af-get-channel hsv 1)
           v (af-get-channel hsv 2)
           ;; Hue mask with circular distance
           hue-dist (af-min (af-abs (af-sub h target-hue))
                            (af-sub 1 (af-abs (af-sub h target-hue))))
           hue-mask (af-lt hue-dist tolerance)
           ;; Saturation and value masks
           sat-mask (af-gt s 0.3)
           val-mask (af-gt v 0.2)
           ;; Combine masks
           mask (af-and hue-mask (af-and sat-mask val-mask))
           ;; Find largest connected component
           labeled (af-regions mask AF_CONNECTIVITY_8)
           centroid (af-moments labeled)]
       centroid))
   ```
   
   ### Batch Image Enhancement
   
   ```clojure
   (defn enhance-images
     [rgb-images brightness saturation]
     (let [hsv (af-rgb2hsv rgb-images)
           [h s v] (split-channels hsv)
           ;; Apply enhancements
           s' (af-clamp (af-mul s saturation) 0 1)
           v' (af-clamp (af-mul v brightness) 0 1)
           hsv' (af-join-channels h s' v')]
       (af-hsv2rgb hsv')))
   ```
   
   ## Comparison with Alternatives
   
   ### vs. RGB Direct Manipulation
   
   **RGB:**
   - Faster (no conversion overhead)
   - Suitable for: blending, compositing, display
   - Difficult for: color adjustment, lighting invariance
   
   **HSV:**
   - Requires conversion (adds 1-2ms)
   - Suitable for: color manipulation, segmentation
   - Intuitive for: brightness/saturation adjustment
   
   ### vs. Other Color Spaces
   
   **HSL (Hue-Saturation-Lightness):**
   - Similar to HSV but different definition of saturation
   - HSL saturation: chroma relative to maximum for given lightness
   - HSV saturation: chroma relative to value
   - Use HSL for: more perceptual uniformity
   - Use HSV for: simpler math, better for segmentation
   
   **LAB (Lightness A B):**
   - Perceptually uniform (equal numeric = equal perceived difference)
   - Better for: color difference metrics, scientific applications
   - More complex conversions (requires XYZ intermediate)
   - HSV better for: real-time applications, intuitive adjustments
   
   **YCbCr:**
   - Separates luminance (Y) from chrominance (Cb, Cr)
   - Better for: video compression, broadcast standards
   - HSV better for: color-based vision algorithms
   
   ## Related ArrayFire Functions
   
   - `af_rgb2gray`: Convert RGB to grayscale
   - `af_gray2rgb`: Convert grayscale to RGB
   - `af_rgb2ycbcr`: Convert RGB to YCbCr color space
   - `af_ycbcr2rgb`: Convert YCbCr to RGB color space
   - `af_color_space`: General color space conversion wrapper
   - `af_clamp`: Clamp array values to range (useful after adjustments)
   - `af_join`: Combine channels into multi-channel array
   - `af_split`: Separate channels from multi-channel array
   
   ## References
   
   - **HSV Color Space**: Smith, A.R. \"Color Gamut Transform Pairs\", SIGGRAPH 1978
   - **Color Vision**: Wandell, B.A. \"Foundations of Vision\", Sinauer Associates, 1995
   - **Computer Graphics**: Foley et al. \"Computer Graphics: Principles and Practice\", 1995
   - **Image Processing**: Gonzalez & Woods \"Digital Image Processing\", 4th Ed., 2018
   - **ArrayFire Documentation**: https://arrayfire.org/docs/group__image__func__hsv2rgb.htm
   - **Color Theory**: Itten, J. \"The Art of Color\", 1961
   
   ## Implementation Notes
   
   ### Backend Support
   
   **Supported:**
   - CPU: Fully supported
   - CUDA: Fully supported, GPU-accelerated
   - OpenCL: Fully supported, GPU-accelerated
   
   **Not Supported:**
   - oneAPI: Returns AF_ERR_NOT_SUPPORTED
   
   ### Thread Safety
   
   - Both functions are thread-safe
   - Can be called concurrently from different threads
   - Each thread should use separate ArrayFire arrays
   
   ### Data Type Support
   
   - **Supported:** f32 (float), f64 (double)
   - **Not supported:** Integer types (u8, s32, etc.)
   - Convert if needed: `(af-cast array AF_F32)`
   
   ### Accuracy
   
   **f32 (float):**
   - Typical round-trip error: <1e-6
   - Sufficient for: display, visualization, most vision tasks
   
   **f64 (double):**
   - Typical round-trip error: <1e-15
   - Recommended for: scientific imaging, professional editing
   
   ### Performance Tips
   
   1. **Reuse allocations:** Pre-allocate output arrays when processing video
   2. **Batch processing:** Combine multiple images in 4D array
   3. **Pipeline in HSV:** Do all adjustments before converting back
   4. **Use f32:** Unless precision critical, f32 is 2× faster
   5. **Async execution:** Queue multiple conversions for overlap"
  (:require [coffi.ffi :as ffi :refer [defcfn]]
            [coffi.mem :as mem]
            [org.soulspace.arrayfire.ffi.loader]))

;; af_err af_hsv2rgb(af_array* out, const af_array in)
(defcfn af-hsv2rgb
  "Convert image from HSV (Hue-Saturation-Value) to RGB (Red-Green-Blue) color space.
   
   Parameters:
   - out: out pointer for RGB array (af_array, 3D with 3 channels)
   - in: input HSV array (af_array, 3D with 3 channels, f32 or f64, values in [0,1])
   
   Returns:
   ArrayFire error code (0=success)"
  "af_hsv2rgb" [::mem/pointer ::mem/pointer] ::mem/int)

;; af_err af_rgb2hsv(af_array* out, const af_array in)
(defcfn af-rgb2hsv
  "Convert image from RGB (Red-Green-Blue) to HSV (Hue-Saturation-Value) color space.
   
   Parameters:
   - out: out pointer for HSV array (af_array, 3D with 3 channels)
   - in: input RGB array (af_array, 3D with 3 channels, f32 or f64, values in [0,1])
   
   Returns:
   ArrayFire error code (0=success)"
  "af_rgb2hsv" [::mem/pointer ::mem/pointer] ::mem/int)
