(ns leiningen.skummet
  (:require [clojure.java.shell :as sh]
            [clojure.java.io :as jio]
            [leiningen.core.project :as pr]
            [leiningen.core.eval :as eval]
            [leiningen.core.main :as main]
            [leiningen.core.classpath :as cp]
            leiningen.clean
            leiningen.compile
            leiningen.run
            [leiningen.jar :as jar]
            leiningen.uberjar)
  (:import (java.io File FileOutputStream PrintWriter)
           (java.util.regex Pattern)
           (java.util.zip ZipFile ZipOutputStream ZipEntry)
           (org.apache.commons.io.output CloseShieldOutputStream)))

(defn eval-in-project
  ([project form init]
   (eval/prep project)
   ;; (println "GLOBAL VARS" (:global-vars project))
   (eval/eval-in project
                 `(do ~@(map (fn [[k v]] `(set! ~k ~v)) (:global-vars project))
                      ~init
                      ~@(:injections project)
                      ~form)))
  ([project form] (eval-in-project project form nil)))

(defn- make-jar [project]
  (jar/write-jar project (jar/get-jar-filename project)
                 (#'jar/filespecs (dissoc project :java-source-paths))))

(defn- uberjar [project]
  (let [standalone-filename (jar/get-jar-filename project :standalone)]
    (with-open [out (-> standalone-filename
                        (FileOutputStream.)
                        (ZipOutputStream.))]
      (let [jar (jar/get-jar-filename project)
            whitelisted (select-keys project jar/whitelist-keys)
            project (merge project whitelisted)
            deps (->> (cp/resolve-dependencies :dependencies project)
                      (filter #(.endsWith (.getName %) ".jar")))
            jars (cons (jio/file jar) deps)]
        (leiningen.uberjar/write-components project jars out)))
    (main/info "Created" standalone-filename)
    standalone-filename))

(defn skummet
  [project & [subtask & args]]
  (let [project project ;(pr/read "/tmp/ring-buffer/project.clj")
        ]
    (cond
      (= subtask "compile")
      (do
        (leiningen.clean/clean project)
        (if-let [namespaces (seq (leiningen.compile/stale-namespaces project))]
          (let [form `(let [lean-var?# (fn [var#] (not (#{~@(:skummet-skip-vars project)}
                                                       (str var#))))]
                        (binding [~'clojure.core/*lean-var?* lean-var?#
                                  ~'clojure.core/*lean-compile* true

                                  ~'clojure.core/*compiler-options*
                                  {:elide-meta [:doc :file :line :added :arglists
                                                :column :static :author :added :dynamic
                                                :inline :declared :private]}]
                          (clojure.lang.RT/resetID)
                          (doseq [namespace# '~namespaces]
                            (println "Compiling" namespace#)
                            (compile namespace#))))
                project (-> project
                            (update-in [:prep-tasks] (partial remove #{"compile"}))
                            (assoc :jvm-opts ["-Dclojure.compile.ignore-lean-classes=true"]))]
            (println "Form: " form)
            (try (eval-in-project project form)
                 (catch Exception e
                   (main/abort "Compilation failed:" (.getMessage e)))))
          (main/debug "All namespaces already AOT compiled.")))

      (= subtask "run")
      (let [deps-line (->> (cp/resolve-dependencies :dependencies project)
                           (cons (jio/file (:target-path project) "classes"))
                           (interpose ":") (apply str))]
        (apply eval/sh "java" "-cp" deps-line (str (:main project)) args))

      (= subtask "jar")
      (do (make-jar project)
          (uberjar project))

      :else
      (main/abort "Wrong subtask:" subtask))))
